/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.pagesubscription.page.notifier;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import org.ametys.core.ui.mail.StandardMailBodyHelper;
import org.ametys.core.ui.mail.StandardMailBodyHelper.MailBodyBuilder;
import org.ametys.core.user.User;
import org.ametys.core.util.URIUtils;
import org.ametys.plugins.explorer.ObservationConstants;
import org.ametys.plugins.explorer.resources.Resource;
import org.ametys.plugins.repository.activities.Activity;
import org.ametys.plugins.repository.activities.ActivityType;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.web.activities.PageResourcesUpdatedActivityType;
import org.ametys.web.repository.page.Page;

/**
 * Notifier for {@link PageResourcesUpdatedActivityType}
 */
public class PageSubscriptionResourcesUpdatedNotifier extends AbstractPageSubscriptionNotifier
{
    private static final int __MAX_RESOURCE_LINK = 5;
    
    @Override
    public boolean supports(ActivityType activityType)
    {
        return activityType instanceof PageResourcesUpdatedActivityType;
    }
    
    @Override
    protected String getSiteParameterId()
    {
        return "page-subscription-attachments";
    }
    
    @Override
    protected I18nizableText _getMailSubject(Activity activity, Page page)
    {
        List<String> i18nparam = new ArrayList<>();
        i18nparam.add(page.getTitle()); // {0}
        
        String i18nKey = null;
        switch (activity.getEventType())
        {
            case ObservationConstants.EVENT_RESOURCE_CREATED:
                String[] resourceIds = activity.getValue(PageResourcesUpdatedActivityType.RESOURCES_IDS);
                int size = resourceIds.length;
                if (size > 1)
                {
                    i18nKey = "PLUGINS_PAGE_SUBSCRIBE_NOTIFICATION_MAIL_ADD_RESOURCES_SUBJECT";
                    i18nparam.add(String.valueOf(size)); // {1}
                }
                else
                {
                    i18nKey = "PLUGINS_PAGE_SUBSCRIBE_NOTIFICATION_MAIL_ADD_RESOURCE_SUBJECT";
                }
                break;
            case ObservationConstants.EVENT_RESOURCE_UPDATED:
                i18nKey = "PLUGINS_PAGE_SUBSCRIBE_NOTIFICATION_MAIL_UPDATE_RESOURCE_SUBJECT";
                break;
            case ObservationConstants.EVENT_RESOURCE_DELETED:
                i18nKey = "PLUGINS_PAGE_SUBSCRIBE_NOTIFICATION_MAIL_DELETE_RESOURCE_SUBJECT";
                break;
            default:
                break;
        }

        return new I18nizableText("plugin.page-subscription", i18nKey, i18nparam);
    }

    @Override
    protected String _getMailHtmlBody(Activity activity, String language, Page page, User author) throws IOException
    {
        String pageUrl = _getAbsolutePageUrl(page);
        String pageTitle = page.getTitle();
        String siteTitle = page.getSite().getTitle();

        MailBodyBuilder bodyBuilder = StandardMailBodyHelper.newHTMLBody();
        
        switch (activity.getEventType())
        {
            case ObservationConstants.EVENT_RESOURCE_CREATED:
                String[] resourceIds = activity.getValue(PageResourcesUpdatedActivityType.RESOURCES_IDS);
                int size = resourceIds.length;
                if (size > 1)
                {
                    bodyBuilder.withTitle(new I18nizableText("plugin.page-subscription", "PLUGINS_PAGE_SUBSCRIBE_NOTIFICATION_MAIL_ADD_RESOURCES_BODY_TITLE", List.of(String.valueOf(size))));
                    bodyBuilder.withMessage(new I18nizableText("plugin.page-subscription", "PLUGINS_PAGE_SUBSCRIBE_NOTIFICATION_MAIL_ADD_RESOURCES_BODY", List.of(String.valueOf(size), pageTitle, siteTitle)));
                    bodyBuilder.withDetails(new I18nizableText("plugin.page-subscription", "PLUGINS_PAGE_SUBSCRIBE_NOTIFICATION_MAIL_ADD_RESOURCES_DETAILS_TITLE"), _getAbsoluteResourcesUrl(resourceIds, page, language), false);
                }
                else
                {
                    Resource resource = _resolver.resolveById(resourceIds[0]);
                    String resourceName = resource.getName();
                    String resourceUrl = _getAbsoluteResourceUrl(resource, page);
                    
                    bodyBuilder.withTitle(new I18nizableText("plugin.page-subscription", "PLUGINS_PAGE_SUBSCRIBE_NOTIFICATION_MAIL_ADD_RESOURCE_BODY_TITLE", List.of(resourceName)));
                    bodyBuilder.addMessage(new I18nizableText("plugin.page-subscription", "PLUGINS_PAGE_SUBSCRIBE_NOTIFICATION_MAIL_ADD_RESOURCE_BODY", List.of(resourceName, pageTitle, siteTitle)));
                    bodyBuilder.addMessage(new I18nizableText("plugin.page-subscription", "PLUGINS_PAGE_SUBSCRIBE_NOTIFICATION_MAIL_RESOURCE_DOWNLOAD", List.of(resourceUrl)));
                }
                break;

            case ObservationConstants.EVENT_RESOURCE_UPDATED:
                String[] updatedResourceIds = activity.getValue(PageResourcesUpdatedActivityType.RESOURCES_IDS);
                Resource resource = _resolver.resolveById(updatedResourceIds[0]);
                String resourceName = resource.getName();
                String resourceUrl = _getAbsoluteResourceUrl(resource, page);
                
                bodyBuilder.withTitle(new I18nizableText("plugin.page-subscription", "PLUGINS_PAGE_SUBSCRIBE_NOTIFICATION_MAIL_UPDATE_RESOURCE_BODY_TITLE", List.of(resourceName)));
                bodyBuilder.addMessage(new I18nizableText("plugin.page-subscription", "PLUGINS_PAGE_SUBSCRIBE_NOTIFICATION_MAIL_UPDATE_RESOURCE_BODY", List.of(resourceName, pageTitle, siteTitle)));
                bodyBuilder.addMessage(new I18nizableText("plugin.page-subscription", "PLUGINS_PAGE_SUBSCRIBE_NOTIFICATION_MAIL_RESOURCE_DOWNLOAD", List.of(resourceUrl)));
                break;

            case ObservationConstants.EVENT_RESOURCE_DELETED:
                String deletedResourceName = activity.getValue(PageResourcesUpdatedActivityType.RESOURCE_DELETED_NAME);
                bodyBuilder.withTitle(new I18nizableText("plugin.page-subscription", "PLUGINS_PAGE_SUBSCRIBE_NOTIFICATION_MAIL_DELETE_RESOURCE_BODY_TITLE", List.of(deletedResourceName)));
                bodyBuilder.addMessage(new I18nizableText("plugin.page-subscription", "PLUGINS_PAGE_SUBSCRIBE_NOTIFICATION_MAIL_DELETE_RESOURCE_BODY", List.of(deletedResourceName, pageTitle, siteTitle)));
                break;
            default:
                break;
        }
        
        return bodyBuilder
                .withLink(pageUrl, new I18nizableText("plugin.page-subscription", "PLUGINS_PAGE_SUBSCRIBE_NOTIFICATION_MAIL_PAGE_LINK_TITLE"))
                .withLanguage(language)
                .build();
    }
    
    private String _getAbsoluteResourcesUrl(String[] resourceIds, Page page, String language)
    {
        StringBuilder sb = new StringBuilder();

        int nbResources = resourceIds.length;
        sb.append("<ul>");
        for (int i = 0; i < Math.min(nbResources, __MAX_RESOURCE_LINK); i++)
        {
            Resource resource = _resolver.resolveById(resourceIds[i]);
            sb.append("<li>")
                   .append("<a href=\"" + _getAbsoluteResourceUrl(resource, page) + "\">")
                   .append(resource.getName())
                   .append("</a>")
                   .append("</li>");
        }

        int nbOthers = nbResources - __MAX_RESOURCE_LINK;
        if (nbOthers > 0)
        {
            I18nizableText moreLinks = new I18nizableText("plugin.page-subscription", "PLUGINS_PAGE_SUBSCRIBE_NOTIFICATION_MAIL_ADD_RESOURCES_MORE_LINKS", List.of(String.valueOf(nbOthers)));
            
            sb.append("<li>")
                .append(_i18nUtils.translate(moreLinks, language))
                .append("</li>");
        }
        
        sb.append("</ul>");

        return sb.toString();
    }

    private String _getAbsoluteResourceUrl(Resource resource, Page page)
    {
        return new StringBuilder().append(page.getSite().getUrl()).append("/_page-attachment").append("?id=").append(URIUtils.encodeParameter(resource.getId())).append(
                "&download=true").toString();
    }
}
