/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.pagesubscription.type;

import java.time.ZonedDateTime;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.lang3.StringUtils;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.core.group.GroupIdentity;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.pagesubscription.BroadcastChannelHelper.BroadcastChannel;
import org.ametys.plugins.pagesubscription.FrequencyHelper.Frequency;
import org.ametys.plugins.pagesubscription.Subscription;
import org.ametys.plugins.pagesubscription.context.SubscriptionContext;
import org.ametys.web.repository.site.Site;

/**
 * Interface for a type of subscription
 * @param <C> the context for the subscription type
 * @param <T> the targeted element of the subscription
 */
public interface SubscriptionType<C extends SubscriptionContext, T>
{
    /**
     * Get the id of the subscription type
     * @return the id of the subscription type
     */
    public String getId();
    
    /**
     * Get the target of the subscription.
     * @param subscription the subscription
     * @return the target
     */
    public T getTarget(Subscription subscription);
    
    /**
     * Get all subscribers for a given frequency and broadcast channel, subscribers from a group subscription including
     * @param site the site
     * @param frequency the frequency
     * @param broadcastChannel the broadcast channel
     * @param context the context
     * @return the subscribers
     */
    public Set<UserIdentity> getSubscribers(Site site, Frequency frequency, BroadcastChannel broadcastChannel, C context);
    
    /**
     * Get the subscribers from a subscription
     * @param subscription the subscription
     * @return the set of subscribers
     */
    public Set<UserIdentity> getSubscribers(Subscription subscription);
    
    /**
     * Subscribe an user
     * @param site the site
     * @param user the user to subscribe
     * @param frequency the frequency
     * @param broadcastChannels the list of broadcast channel
     * @param context the context
     * @return the subscription
     * @throws Exception if an error occurred
     */
    public Subscription subscribe(Site site, UserIdentity user, Frequency frequency, List<BroadcastChannel> broadcastChannels, C context) throws Exception;
    
    /**
     * Force a user subscription
     * @param site the site
     * @param user the user to subscribe
     * @param frequency the frequency
     * @param forcedTiming the force frequency timing
     * @param broadcastChannels the list of broadcast channel
     * @param context the context
     * @return the subscription
     * @throws Exception if an error occurred
     */
    public Subscription forceSubscription(Site site, UserIdentity user, Frequency frequency, FrequencyTiming forcedTiming, List<BroadcastChannel> broadcastChannels, C context) throws Exception;
    
    /**
     * Subscribe a group
     * @param site the site
     * @param group the group to subscribe
     * @param frequency the frequency
     * @param broadcastChannels the list of broadcast channel
     * @param context the context
     * @return the subscription
     * @throws Exception if an error occurred
     */
    public Subscription subscribe(Site site, GroupIdentity group, Frequency frequency, List<BroadcastChannel> broadcastChannels, C context) throws Exception;
    
    /**
     * Force a group subscription
     * @param site the site
     * @param group the group to subscribe
     * @param frequency the frequency
     * @param forcedTiming the force frequency timing
     * @param broadcastChannels the list of broadcast channel
     * @param context the context
     * @return the subscription
     * @throws Exception if an error occurred
     */
    public Subscription forceSubscription(Site site, GroupIdentity group, Frequency frequency, FrequencyTiming forcedTiming, List<BroadcastChannel> broadcastChannels, C context) throws Exception;
    
    
    /**
     * Edit a subscription
     * @param subscription the subscription to edit
     * @param frequency the frequency
     * @param broadcastChannels the list of broadcast channel
     * @param context the context
     * @throws Exception if an error occurred
     */
    public void editSubscription(Subscription subscription, Frequency frequency, List<BroadcastChannel> broadcastChannels, C context) throws Exception;
    
    /**
     * Edit a force subscription
     * @param subscription the subscription to edit
     * @param frequency the frequency
     * @param forcedTiming the force frequency timing
     * @param broadcastChannels the list of broadcast channel
     * @param context the context
     * @throws Exception if an error occurred
     */
    public void editForceSubscription(Subscription subscription, Frequency frequency, List<BroadcastChannel> broadcastChannels, FrequencyTiming forcedTiming, C context) throws Exception;
    
    /**
     * Unsubscribe a user from all their subscriptions on the site
     * (including forced subscription directly on the user).
     * @param site the site
     * @param user the user to unsubscribe
     * @return the number of deleted subscriptions
     * @throws Exception if an error occurred
     */
    public long unsubscribeAll(Site site, UserIdentity user) throws Exception;
    
    /**
     * Unsubscribe a user from all their subscriptions on the context
     * (including forced subscription directly on the user).
     * @param site the site
     * @param user the user to unsubscribe
     * @param context the context
     * @return the number of deleted subscriptions
     * @throws Exception if an error occurred
     */
    public long unsubscribe(Site site, UserIdentity user, C context) throws Exception;
    
    /**
     * Unsubscribe a group from all their subscriptions on the site
     * (including forced subscription directly on the group).
     * @param site the site
     * @param group the group to unsubscribe
     * @return the number of deleted subscriptions
     * @throws Exception if an error occurred
     */
    public long unsubscribeAll(Site site, GroupIdentity group) throws Exception;

    /**
     * Unsubscribe a group from all their subscriptions on the context
     * (including forced subscription directly on the group).
     * @param site the site
     * @param group the group to unsubscribe
     * @param context the context
     * @return the number of deleted subscriptions
     * @throws Exception if an error occurred
     */
    public long unsubscribe(Site site, GroupIdentity group, C context) throws Exception;
    
    /**
     * Unsubscribe
     * @param subscription the subscription to unsubscribe
     * @throws Exception if an error occurred
     */
    public void unsubscribe(Subscription subscription) throws Exception;
    
    /**
     * Subscription to JSON
     * @param subscription the subscription
     * @return the subscription as JSON
     */
    public Map<String, Object> subscriptionToJSON(Subscription subscription);
    
    /**
     * SAX a subscription
     * @param contentHandler the content handler to sax into
     * @param subscription the subscription
     * @throws SAXException if a error occured while saxing
     */
    public void saxSubscription(ContentHandler contentHandler, Subscription subscription) throws SAXException;
    
    /**
     * Get the last read date of the subscription for the given user
     * @param subscription the subscription
     * @param user the user
     * @return the last read date of the subscription
     */
    public ZonedDateTime getLastReadDate(Subscription subscription, UserIdentity user);
    
    /**
     * Mark the subscription as read for the given user
     * @param subscription the subscription
     * @param user the user
     */
    public void markAsRead(Subscription subscription, UserIdentity user);
    
    /**
     * The day and time of the frequency
     * @param day the day
     * @param time the time as "HH:mm"
     */
    public record FrequencyTiming(Long day, String time) {
        /**
         * Get the hour of notification
         * @return the hour
         */
        public long hour()
        {
            return StringUtils.isNotEmpty(time()) ? Long.valueOf(StringUtils.substringBefore(time(), ":")) : -1;
        }
        
        /**
         * Get the minutes of notification
         * @return the minutes
         */
        public long minutes()
        {
            return StringUtils.isNotEmpty(time()) ? Long.valueOf(StringUtils.substringAfter(time(), ":")) : -1;
        }
    }
}
