/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.pagesubscription.tracking;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Objects;
import java.util.Set;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.core.group.Group;
import org.ametys.core.group.GroupManager;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.pagesubscription.Subscription;
import org.ametys.plugins.pagesubscription.type.PageSubscriptionType;
import org.ametys.plugins.pagesubscription.type.SubscriptionTypeExtensionPoint;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.webanalytics.matomo.tracking.AbstractMatomoEventTrackingProvider;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.site.Site;

/**
 * The matomo tracking provider for the page subscription
 */
public class PageSubscriptionMatomoTrackingProvider extends AbstractMatomoEventTrackingProvider implements Serviceable
{
    /** The subscription type extension point */
    protected SubscriptionTypeExtensionPoint _subscriptionTypeEP;
    
    /** The group manager */
    protected GroupManager _groupManager;
    
    /** The ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _subscriptionTypeEP = (SubscriptionTypeExtensionPoint) manager.lookup(SubscriptionTypeExtensionPoint.ROLE);
        _groupManager = (GroupManager) manager.lookup(GroupManager.ROLE);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
    }
    
    @Override
    protected List<MatomoEvent> getEvents(Site site) 
    {
        List<MatomoEvent> requests = new ArrayList<>();
        for (Entry<String, Set<UserIdentity>> entry: _getSubscribers(site).entrySet())
        {
            String pageId = entry.getKey();
            Page page = _silentlyResolve(pageId);
            if (page != null)
            {
                requests.add(new MatomoEvent(
                        page.getSitemapName() + "/" + page.getPathInSitemap(),
                        entry.getValue().size()
                        ));
            }
        }
        
        return requests;
    }
    
    private Page _silentlyResolve(String pageId)
    {
        try
        {
            return _resolver.resolveById(pageId);
        }
        catch (Exception e) 
        {
            getLogger().warn("Cannot track subscription for the page with id '{}'. It does not seem to exist.", pageId, e);
            return null;
        }
    }
    
    private Map<String, Set<UserIdentity>> _getSubscribers(Site site)
    {
        PageSubscriptionType subscriptionType = (PageSubscriptionType) _subscriptionTypeEP.getExtension(PageSubscriptionType.ID);

        Map<String, Set<UserIdentity>> subscribers = new HashMap<>();
        for (Subscription subscription : subscriptionType.getSubscriptions(site, null, null, null))
        {
            String pageId = subscriptionType.getTarget(subscription);
            
            Set<UserIdentity> users = subscribers.computeIfAbsent(pageId, __ -> new HashSet<>());
            subscription.getSubscriber().ifPresent(users::add);
            
            subscription.getSubscribersGroup()
                .map(_groupManager::getGroup)
                .filter(Objects::nonNull)
                .map(Group::getUsers)
                .ifPresent(users::addAll);
        }
        
        return subscribers;
    }
    
}
