/*
 *  Copyright 2014 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.pagesubscription.page;

import java.io.IOException;
import java.util.regex.Pattern;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.generation.ServiceableGenerator;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.xml.sax.SAXException;

import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.User;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.user.UserManager;
import org.ametys.core.util.mail.SendMailHelper;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.web.repository.page.Page;

/**
 * Generate user info about a page
 */
public class InfoSubscriberGenerator extends ServiceableGenerator
{
    /** The pattern to check emails */
    protected static final Pattern EMAIL_VALIDATOR = SendMailHelper.EMAIL_VALIDATION;
    
    /** The pattern to check text input */
    protected static final Pattern TEXT_VALIDATOR = Pattern.compile("^\\s*$");
    
    /** Repository content */
    protected AmetysObjectResolver _resolver;
    
    /** FO user manager */
    protected UserManager _foUserManager;
    
    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;
    
    /** The subscription dao */
    protected PageSubscriptionDAO _subscriptionDAO;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _subscriptionDAO = (PageSubscriptionDAO) smanager.lookup(PageSubscriptionDAO.ROLE);
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
        _foUserManager = (UserManager) smanager.lookup(UserManager.ROLE);
        _currentUserProvider = (CurrentUserProvider) smanager.lookup(CurrentUserProvider.ROLE);
    }

    public void generate() throws IOException, SAXException, ProcessingException
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        
        UserIdentity userIdentity = _currentUserProvider.getUser();
        
        contentHandler.startDocument();
        AttributesImpl attr = new AttributesImpl();
        if (userIdentity == null)
        {
            attr.addCDATAAttribute("isLogged", "false");
            XMLUtils.createElement(contentHandler, "user", attr);
        }
        else
        {
            User user = _foUserManager.getUser(userIdentity.getPopulationId(), userIdentity.getLogin());
            
            attr.addCDATAAttribute("isLogged", "true");
            String email = user.getEmail();
          
            String pageId = request.getParameter("page-id");
            if (pageId == null) 
            {
                throw new IllegalArgumentException("Unable to subscribe or unsubscribe to the page: cannot determine the current page");
            }
            else
            {
                Page page = _resolver.resolveById(pageId);
                boolean isSubscriber = _subscriptionDAO.isSubscriber(page, email);
                attr.addCDATAAttribute("isRegistered", Boolean.toString(isSubscriber));
            }
            
            XMLUtils.createElement(contentHandler, "user", attr, email);    
        }
        contentHandler.endDocument();
    }
}
