/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.pagesubscription.tracking;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Objects;
import java.util.Set;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.cms.tag.CMSTag;
import org.ametys.core.group.Group;
import org.ametys.core.group.GroupManager;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.util.I18nUtils;
import org.ametys.plugins.pagesubscription.Subscription;
import org.ametys.plugins.pagesubscription.type.SubscriptionTypeExtensionPoint;
import org.ametys.plugins.pagesubscription.type.TagSubscriptionType;
import org.ametys.plugins.webanalytics.matomo.tracking.AbstractMatomoEventTrackingProvider;
import org.ametys.web.repository.site.Site;

/**
 * The matomo tracking provider for the tag subscription
 */
public class TagSubscriptionMatomoTrackingProvider extends AbstractMatomoEventTrackingProvider implements Serviceable
{
    /** The subscription type extension point */
    protected SubscriptionTypeExtensionPoint _subscriptionTypeEP;
    
    /** The group manager */
    protected GroupManager _groupManager;
    
    /** The i18n utils */
    protected I18nUtils _i18nUtils;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _subscriptionTypeEP = (SubscriptionTypeExtensionPoint) manager.lookup(SubscriptionTypeExtensionPoint.ROLE);
        _groupManager = (GroupManager) manager.lookup(GroupManager.ROLE);
        _i18nUtils = (I18nUtils) manager.lookup(I18nUtils.ROLE);
    }
    
    @Override
    protected List<MatomoEvent> getEvents(Site site) 
    {
        List<MatomoEvent> requests = new ArrayList<>();
        for (Entry<CMSTag, Set<UserIdentity>> entry: _getSubscribers(site).entrySet())
        {
            CMSTag tag = entry.getKey();
            requests.add(new MatomoEvent(
                _i18nUtils.translate(tag.getTitle()), // Accept to take the server language for the matomo statistics
                entry.getValue().size()
            ));
        }
        
        return requests;
    }
    
    private Map<CMSTag, Set<UserIdentity>> _getSubscribers(Site site)
    {
        TagSubscriptionType subscriptionType = (TagSubscriptionType) _subscriptionTypeEP.getExtension(TagSubscriptionType.ID);

        Map<CMSTag, Set<UserIdentity>> subscribers = new HashMap<>();
        for (Subscription subscription : subscriptionType.getSubscriptions(site, null, null, null))
        {
            CMSTag tag = subscriptionType.getTarget(subscription);
            
            Set<UserIdentity> users = subscribers.computeIfAbsent(tag, __ -> new HashSet<>());
            subscription.getSubscriber().ifPresent(users::add);
            
            subscription.getSubscribersGroup()
                .map(_groupManager::getGroup)
                .filter(Objects::nonNull)
                .map(Group::getUsers)
                .ifPresent(users::addAll);
        }
        
        return subscribers;
    }
    
}
