/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.pagesubscription.type;

import java.util.Map;

import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.cms.transformation.xslt.ResolveURIComponent;
import org.ametys.plugins.pagesubscription.BroadcastChannelHelper.BroadcastChannel;
import org.ametys.plugins.pagesubscription.FrequencyHelper;
import org.ametys.plugins.pagesubscription.FrequencyHelper.Frequency;
import org.ametys.plugins.pagesubscription.Subscription;
import org.ametys.plugins.pagesubscription.context.PageSubscriptionContext;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.StringExpression;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.web.repository.page.Page;

/**
 * Page subscription type
 */
public class PageSubscriptionType extends AbstractSubscriptionType<PageSubscriptionContext, String>
{
    /** The id of the extension */
    public static final String ID = "subscription.page";

    /** Data name for the page of the subscription */
    public static final String PAGE = "page";
    
    /** Data name for the sitemap page of the subscription */
    public static final String SITEMAP = "sitemap";
    
    public String getTarget(Subscription subscription)
    {
        return subscription.getValue(PAGE);
    }
    
    @Override
    protected void setAdditionalData(Subscription subscription, PageSubscriptionContext context)
    {
        Page page = context.getPage();
        subscription.setValue(PAGE, page.getId());
        subscription.setValue(SITEMAP, page.getSitemapName());
    }
    
    @Override
    protected boolean isSubscriptionValid(Subscription subscription)
    {
        // Check is the page still exist
        String pageId = subscription.getValue(PAGE);
        return StringUtils.isNotBlank(pageId)
                 ? _resolver.hasAmetysObjectForId(pageId)
                 : false;
    }
    
    @Override
    protected Expression getAdditionalFilterExpression(PageSubscriptionContext context)
    {
        Page page = context.getPage();
        if (page != null)
        {
            return new StringExpression(PAGE, Operator.EQ, page.getId());
        }
        return null;
    }
    
    @Override
    public Map<String, Object> subscriptionToJSON(Subscription subscription)
    {
        Map<String, Object> subscriptionToJSON = super.subscriptionToJSON(subscription);
        
        String pageId = subscription.getValue(PAGE);
        try 
        {
            Page page = _resolver.resolveById(pageId);
            subscriptionToJSON.put("page", Map.of(
                "label", page.getTitle(),
                "id", page.getId(),
                "url", ResolveURIComponent.resolve("page", pageId, false)
            ));
            
        }
        catch (Exception e) 
        {
            getLogger().error("An error occurred getting page for id '{}'", pageId, e);
        }
        
        return subscriptionToJSON;
    }
    
    @Override
    protected Map<String, Object> _frequencyToJson(Subscription subscription)
    {
        Map<String, Object> frequency2json = super._frequencyToJson(subscription);
        
        frequency2json.put("label", _getPageSubscriptionFrequencyLabel(subscription));
        frequency2json.put("smartLabel", _getPageSubscriptionFrequencySmartLabel(subscription));
        
        return frequency2json;
    }
    
    @Override
    protected void _saxAdditionalData(ContentHandler contentHandler, Subscription subscription) throws SAXException
    {
        super._saxAdditionalData(contentHandler, subscription);
        
        String pageId = subscription.getValue(PAGE);
        
        try 
        {
            Page page = _resolver.resolveById(pageId);
            AttributesImpl attrsPage = new AttributesImpl();
            attrsPage.addCDATAAttribute("id", pageId);
            XMLUtils.createElement(contentHandler, "page", attrsPage, page.getTitle());
            
        }
        catch (UnknownAmetysObjectException e) 
        {
            getLogger().error("Unknown page for id '{}'", pageId, e);
        }
    }
    
    @Override
    protected void _saxFrequency(ContentHandler contentHandler, Subscription subscription) throws SAXException
    {
        Frequency frequency = subscription.getFrequency();

        AttributesImpl attrs = new AttributesImpl();
        attrs.addCDATAAttribute("name", frequency.name());
        
        XMLUtils.startElement(contentHandler, "frequency", attrs);
        
        // Override label and smart label for page
        _getPageSubscriptionFrequencyLabel(subscription).toSAX(contentHandler, "label");
        _getPageSubscriptionFrequencySmartLabel(subscription).toSAX(contentHandler, "smartLabel");
        FrequencyHelper.getFullLabel(frequency, subscription.getBroadcastChannels());
        
        XMLUtils.endElement(contentHandler, "frequency");
    }
    
    /**
     * Get the frequency label for a page subscription
     * @param pageSubscription the page subscription
     * @return the frequency label
     */
    private I18nizableText _getPageSubscriptionFrequencyLabel(Subscription pageSubscription)
    {
        return pageSubscription.getBroadcastChannels().contains(BroadcastChannel.MAIL)
                ? FrequencyHelper.getLabel(pageSubscription.getFrequency())
                : new I18nizableText("plugin.page-subscription", "PLUGINS_PAGE_SUBSCRIBE_USER_SUBSCRIPTIONS_FREQUENCY_NO_MAIL");
    }
    
    /**
     * Get the frequency smart label for a page subscription
     * @param pageSubscription the page subscription
     * @return the frequency smart label
     */
    private I18nizableText _getPageSubscriptionFrequencySmartLabel(Subscription pageSubscription)
    {
        return pageSubscription.getBroadcastChannels().contains(BroadcastChannel.MAIL)
                ? FrequencyHelper.getSmartLabel(pageSubscription.getFrequency())
                : new I18nizableText("plugin.page-subscription", "PLUGINS_PAGE_SUBSCRIBE_USER_SUBSCRIPTIONS_FREQUENCY_NO_MAIL");
    }
    
    @Override
    protected String getUserPreferenceContextId(Subscription subscription)
    {
        return getTarget(subscription);
    }
}
