/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.pagesubscription.generator;

import java.io.IOException;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.generation.ServiceableGenerator;
import org.apache.cocoon.xml.XMLUtils;
import org.xml.sax.SAXException;

import org.ametys.core.user.CurrentUserProvider;
import org.ametys.plugins.pagesubscription.FrequencyHelper;
import org.ametys.plugins.pagesubscription.Subscription;
import org.ametys.plugins.pagesubscription.context.PageSubscriptionContext;
import org.ametys.plugins.pagesubscription.type.PageSubscriptionType;
import org.ametys.plugins.pagesubscription.type.SubscriptionTypeExtensionPoint;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.web.WebHelper;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.site.SiteManager;

/**
 * SAX user subscriptions
 */
public class UserPageSubscriptionsGenerator extends ServiceableGenerator
{
    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;
    
    /** The site manager */
    protected SiteManager _siteManager;
    
    /** The subscription type extension point */
    protected SubscriptionTypeExtensionPoint _subscriptionTypeEP;
    
    /** The ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(manager);
        _currentUserProvider = (CurrentUserProvider) smanager.lookup(CurrentUserProvider.ROLE);
        _subscriptionTypeEP = (SubscriptionTypeExtensionPoint) smanager.lookup(SubscriptionTypeExtensionPoint.ROLE);
        _siteManager = (SiteManager) smanager.lookup(SiteManager.ROLE);
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
    }
    
    @Override
    public void generate() throws IOException, SAXException, ProcessingException
    {
        contentHandler.startDocument();
        XMLUtils.startElement(contentHandler, "subscriptions");

        FrequencyHelper.saxFrequencies(contentHandler);
        
        Request request = ObjectModelHelper.getRequest(objectModel);
        String siteName = WebHelper.getSiteName(request);
        PageSubscriptionType subscriptionType = (PageSubscriptionType) _subscriptionTypeEP.getExtension(PageSubscriptionType.ID);
        
        Site site = _siteManager.getSite(siteName);
        for (Subscription subscription : subscriptionType.getUserSubscriptions(site, _currentUserProvider.getUser(), false, PageSubscriptionContext.newInstance()))
        {
            subscriptionType.saxSubscription(contentHandler, subscription);
        }

        XMLUtils.endElement(contentHandler, "subscriptions");
        contentHandler.endDocument();
    }
    
}
