/*
 *  Copyright 2013 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.queriesdirectory.actions;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Predicate;
import java.util.stream.Stream;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.acting.ServiceableAction;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.SourceResolver;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;

import org.ametys.core.cocoon.JSonReader;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.queriesdirectory.Query;
import org.ametys.plugins.queriesdirectory.Query.QueryProfile;
import org.ametys.plugins.queriesdirectory.QueryContainer;
import org.ametys.plugins.queriesdirectory.QueryDAO;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;

import com.google.common.base.Predicates;

/**
 * SAX queries
 *
 */
public class GetQueriesAction extends ServiceableAction
{
    private CurrentUserProvider _userProvider;
    private AmetysObjectResolver _resolver;
    private QueryDAO _queryDAO;
    
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _userProvider = (CurrentUserProvider) serviceManager.lookup(CurrentUserProvider.ROLE);
        _resolver = (AmetysObjectResolver) serviceManager.lookup(AmetysObjectResolver.ROLE);
        _queryDAO = (QueryDAO) serviceManager.lookup(QueryDAO.ROLE);
    }
    
    public Map act(Redirector redirector, SourceResolver resolver, Map objectModel, String source, Parameters parameters) throws Exception
    {
        @SuppressWarnings("unchecked")
        Map<String, Object> jsParameters = (Map<String, Object>) objectModel.get(ObjectModelHelper.PARENT_CONTEXT);
        
        String profileId = StringUtils.defaultIfEmpty((String) jsParameters.get("profile"), "read_access");
        
        @SuppressWarnings("unchecked")
        List<String> acceptedQueryTypes = (List<String>) jsParameters.get("acceptedQueryTypes");
        
        if (!"read_access".equals(profileId) && !"write_access".equals(profileId) && !"right_access".equals(profileId))
        {
            throw new IllegalArgumentException("Unexpected profile identifier : " + profileId);
        }
        
        QueryProfile profile = QueryProfile.valueOf(profileId.toUpperCase());
        QueryContainer parentContainer = _getParentContainer(jsParameters);
        Boolean onlyContainers = (Boolean) jsParameters.getOrDefault("onlyContainers", Boolean.FALSE);
        Boolean onlyQueries = onlyContainers ? false : (Boolean) jsParameters.getOrDefault("onlyQueries", Boolean.FALSE);
        Boolean testIfEmptyContainer = (Boolean) jsParameters.getOrDefault("testIfEmptyContainer", Boolean.FALSE);
        Boolean allDescendants = (Boolean) jsParameters.getOrDefault("allDescendants", Boolean.FALSE);
        
        UserIdentity user = _userProvider.getUser();
        
        List<Map<String, Object>> nodes = new ArrayList<>();
        Stream<Query> childQueriesStream = onlyContainers
                ? Stream.empty()
                : _getChildQueries(parentContainer, !allDescendants, acceptedQueryTypes, profile, user);
        childQueriesStream
                .map(_queryDAO::getQueryProperties)
                .forEach(nodes::add);
        
        Stream<Pair<QueryContainer, AdditionalInfoOnContainer>> childContainersStream = onlyQueries
                ? Stream.empty()
                : _getChildQueryContainers(parentContainer, testIfEmptyContainer, acceptedQueryTypes, profile, user);
        
        childContainersStream
                .forEach(p ->
                {
                    QueryContainer queryContainer = p.getLeft();
                    Map<String, Object> properties = _queryDAO.getQueryContainerProperties(queryContainer);
                    AdditionalInfoOnContainer additionalInfo = p.getRight();
                    additionalInfo.fillAdditionalInfo(properties);
                    switch (profile)
                    {
                        case WRITE_ACCESS:
                            if ((boolean) properties.get(QueryDAO.WRITE_ACCESS_PROPERTY) || _queryDAO.hasAnyWritableDescendant(user, queryContainer))
                            {
                                nodes.add(properties);
                            }
                            break;
                        case RIGHT_ACCESS:
                            if ((boolean) properties.get(QueryDAO.EDIT_RIGHTS_ACCESS_PROPERTY) || _queryDAO.hasAnyAssignableDescendant(user, queryContainer))
                            {
                                nodes.add(properties);
                            }
                            break;
                        case READ_ACCESS:
                        default:
                            if ((boolean) properties.get(QueryDAO.READ_ACCESS_PROPERTY) || _queryDAO.hasAnyReadableDescendant(user, queryContainer))
                            {
                                nodes.add(properties);
                            }
                            break;
                    }
                });
        
        Map<String, Object> result = new HashMap<>();
        result.put("queries", nodes);
        
        Request request = ObjectModelHelper.getRequest(objectModel);
        request.setAttribute(JSonReader.OBJECT_TO_READ, result);
        
        return EMPTY_MAP;
    }
    
    private QueryContainer _getParentContainer(Map jsParameters)
    {
        Predicate<String> isNotRoot = Predicates.not(Predicates.equalTo(QueryDAO.ROOT_QUERY_CONTAINER_ID));
        QueryContainer parent = Optional.of("node")
            .map(jsParameters::get)
            .filter(String.class::isInstance)
            .map(String.class::cast)
            .filter(isNotRoot)
            .map(_resolver::<QueryContainer>resolveById)
            .orElse(_queryDAO.getQueriesRootNode()/* root node must exist, so it must be created if it does not exist yet */);
        return parent;
    }
    
    private Stream<Pair<QueryContainer, AdditionalInfoOnContainer>> _getChildQueryContainers(QueryContainer parentContainer, boolean testIfEmptyContainer, List<String> acceptedQueryTypes, QueryProfile profile, UserIdentity user)
    {
        AmetysObjectIterable<QueryContainer> childQueryContainers = _queryDAO.getChildQueryContainers(parentContainer);
        
        Stream<QueryContainer> childQueryContainersStream = childQueryContainers.stream();
        if (testIfEmptyContainer)
        {
            // indicates if container do not have (not necessarily direct) query children (i.e. is empty)
            return childQueryContainersStream
                    .map(ct -> Pair.of(ct, new AdditionalInfoOnContainer(_hasNoDescendant(ct, acceptedQueryTypes, profile, user))));
        }
        else
        {
            return childQueryContainersStream
                    .map(ct -> Pair.of(ct, new AdditionalInfoOnContainer()));
        }
    }
    
    private Boolean _hasNoDescendant(QueryContainer queryContainer, List<String> acceptedQueryTypes, QueryProfile profile, UserIdentity user)
    {
        boolean onlyDirectChildren = false; // in order to test no only direct children, but all descendants
        Stream<Query> childQueries = _getChildQueries(queryContainer, onlyDirectChildren, acceptedQueryTypes, profile, user);
        return !childQueries.findAny().isPresent();
    }
    
    private Stream<Query> _getChildQueries(QueryContainer parentContainer, boolean onlyDirectChildren, List<String> acceptedQueryTypes, QueryProfile profile, UserIdentity user)
    {
        Stream<Query> childQueries;
        switch (profile)
        {
            case WRITE_ACCESS:
                childQueries = _queryDAO.getChildQueriesInWriteAccess(parentContainer, onlyDirectChildren, user, acceptedQueryTypes);
                break;
            case RIGHT_ACCESS:
                childQueries = _queryDAO.getChildQueriesInRightAccess(parentContainer, onlyDirectChildren, user, acceptedQueryTypes);
                break;
            case READ_ACCESS:
            default:
                childQueries = _queryDAO.getChildQueriesInReadAccess(parentContainer, onlyDirectChildren, user, acceptedQueryTypes);
                break;
        }
        
        return childQueries;
    }
    
    private static class AdditionalInfoOnContainer
    {
        private Optional<Boolean> _hasNoDescendant = Optional.empty();
        
        AdditionalInfoOnContainer()
        {
        }
        
        AdditionalInfoOnContainer(boolean hasNoDescendant)
        {
            _hasNoDescendant = Optional.of(hasNoDescendant);
        }
        
        void fillAdditionalInfo(Map<String, Object> props)
        {
            _hasNoDescendant.ifPresent(b -> props.put("hasNoDescendantQuery", b));
        }
    }

}
