/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.queriesdirectory.scripts;

import java.util.HashMap;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.cms.scripts.CmsScriptHandler;
import org.ametys.core.ui.Callable;
import org.ametys.core.util.JSONUtils;
import org.ametys.plugins.queriesdirectory.QueriesDirectoryRightAssignmentContext;
import org.ametys.plugins.queriesdirectory.Query;
import org.ametys.plugins.queriesdirectory.QueryDAO;
import org.ametys.plugins.repository.AmetysObjectResolver;

/**
 * The CmsScriptHandler that only executes scripts for the Queries Directory plugin
 */
public class QueriesDirectoryScriptHandler extends CmsScriptHandler
{
    private AmetysObjectResolver _ametysObjectResover;
    private JSONUtils _jsonUtils;

    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        
        _ametysObjectResover = (AmetysObjectResolver) serviceManager.lookup(AmetysObjectResolver.ROLE);
        _jsonUtils = (JSONUtils) serviceManager.lookup(JSONUtils.ROLE);
    }
    
    /**
     * Execute a script saved in the queries directory.
     * @param queryId The id of the query to execute.
     * @param arguments The arguments to pass to the executeScript method.
     * @return A map of information on the script execution.
     */
    @SuppressWarnings("unchecked")
    @Callable(rights = {Callable.READ_ACCESS, QueryDAO.QUERY_HANDLE_RIGHT_ID}, paramIndex = 0, rightContext = QueriesDirectoryRightAssignmentContext.ID)
    public Map<String, Object> executeScript(String queryId, Map<String, Object> arguments)
    {
        Query query = _ametysObjectResover.resolveById(queryId);
        String queryContentAsString = query.getContent();
        Map<String, Object> queryContentAsMap = _jsonUtils.convertJsonToMap(queryContentAsString);
        Map<String, Object> toolParams = (Map<String, Object>) queryContentAsMap.get("toolParams");
        Map<String, Object> values = (Map<String, Object>) toolParams.get("values");
        String script = (String) values.get("script");
        
        Map<String, Object> modifiedArguments = new HashMap<>(arguments);
        modifiedArguments.put("script", script);
        return super.executeScript(modifiedArguments);
    }
}
