/*
 *  Copyright 2013 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.queriesdirectory;

import javax.jcr.Node;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.core.group.GroupManager;
import org.ametys.plugins.repository.AmetysObjectFactory;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.repository.data.type.ModelItemTypeExtensionPoint;
import org.ametys.plugins.repository.jcr.DefaultAmetysObjectFactory;
import org.ametys.runtime.model.DefaultElementDefinition;
import org.ametys.runtime.model.Model;
import org.ametys.runtime.model.exception.BadItemTypeException;
import org.ametys.runtime.model.exception.UnknownTypeException;
import org.ametys.runtime.model.type.ModelItemTypeConstants;


/**
 * {@link AmetysObjectFactory} for handling {@link Query}s.
 */
public class QueryFactory extends DefaultAmetysObjectFactory
{
    
    /** JCR nodetype for query */
    public static final String QUERY_NODETYPE = RepositoryConstants.NAMESPACE_PREFIX + ":query";
    
    /** Group manager */
    protected GroupManager _groupManager;
    
    /** Content type extension point */
    protected ContentTypeExtensionPoint _cTypeEP;

    /** The Query DAO */
    protected QueryDAO _queryDAO;
    
    /** The Query model */
    protected Model _model;

    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _cTypeEP = (ContentTypeExtensionPoint) manager.lookup(ContentTypeExtensionPoint.ROLE);
        _groupManager = (GroupManager) manager.lookup(GroupManager.ROLE);
        _queryDAO = (QueryDAO) manager.lookup(QueryDAO.ROLE);
    }
    
    @Override
    public Query getAmetysObject(Node node, String parentPath) throws AmetysRepositoryException
    {
        return new Query(node, parentPath, this);
    }
    
    /**
     * Group manager getter.
     * @return The group manager
     */
    GroupManager _getGroupManager()
    {
        return _groupManager;
    }
    
    /**
     * Getter for the content type extension point
     * @return The content type EP.
     */
    ContentTypeExtensionPoint _getContentTypeExtensionPoint()
    {
        return _cTypeEP;
    }
    
    QueryDAO getQueryDAO()
    {
        return _queryDAO;
    }

    /**
     * Retrieves the query model
     * @return the query model
     */
    public Model getModel()
    {
        if (_model == null)
        {
            _model = _createQueryModel();
        }
        return _model;
    }
    
    /**
     * Creates the query model
     * @return the created model
     * @throws AmetysRepositoryException if an error occurs.
     */
    protected Model _createQueryModel() throws AmetysRepositoryException
    {
        try
        {
            String role = ModelItemTypeExtensionPoint.ROLE_MODEL_AWARE_BASIC;
            return Model.of(Query.class.getName(), Query.class.getName(),
                    DefaultElementDefinition.of(Query.TITLE, false, ModelItemTypeConstants.STRING_TYPE_ID, role),
                    DefaultElementDefinition.of(Query.DESCRIPTION, false, ModelItemTypeConstants.STRING_TYPE_ID, role),
                    DefaultElementDefinition.of(Query.DOCUMENTATION, false, ModelItemTypeConstants.STRING_TYPE_ID, role),
                    DefaultElementDefinition.of(Query.CONTENT, false, ModelItemTypeConstants.STRING_TYPE_ID, role),
                    DefaultElementDefinition.of(Query.TYPE, false, ModelItemTypeConstants.STRING_TYPE_ID, role),
                    DefaultElementDefinition.of(Query.CREATIONDATE, false, ModelItemTypeConstants.DATETIME_TYPE_ID, role),
                    DefaultElementDefinition.of(Query.LASTMODIFICATIONDATE, false, ModelItemTypeConstants.DATETIME_TYPE_ID, role),
                    DefaultElementDefinition.of(Query.AUTHOR, false, org.ametys.cms.data.type.ModelItemTypeConstants.USER_ELEMENT_TYPE_ID, role),
                    DefaultElementDefinition.of(Query.CONTRIBUTOR, false, org.ametys.cms.data.type.ModelItemTypeConstants.USER_ELEMENT_TYPE_ID, role));
        }
        catch (UnknownTypeException | BadItemTypeException | ServiceException e)
        {
            throw new AmetysRepositoryException("An error occurred while creating the query model", e);
        }
    }
}
