/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.queriesdirectory.accesscontroller;

import java.util.Collections;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.core.right.AccessController;
import org.ametys.core.right.RightsException;
import org.ametys.plugins.core.impl.right.AbstractHierarchicalAccessController;
import org.ametys.plugins.queriesdirectory.Query;
import org.ametys.plugins.queriesdirectory.QueryContainer;
import org.ametys.plugins.queriesdirectory.QueryDAO;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.i18n.I18nizableTextParameter;

/**
 * {@link AccessController} for a {@link Query}
 */
public class QueryAccessController extends AbstractHierarchicalAccessController<AmetysObject>
{
    /** the queries directory context category */
    public static final I18nizableText QUERIES_DIRECTORY_CONTEXT_CATEGORY = new I18nizableText("plugin.queries-directory", "PLUGINS_QUERIESDIRECTORY_RIGHTS_CATEGORY");
    /** The query DAO */
    protected QueryDAO _queryDAO;

    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _queryDAO = (QueryDAO) manager.lookup(QueryDAO.ROLE);
    }
    
    public boolean supports(Object object)
    {
        return object instanceof Query || object instanceof QueryContainer;
    }

    @Override
    protected Set<AmetysObject> _getParents(AmetysObject object)
    {
        AmetysObject parent = object.getParent();
        if (supports(parent))
        {
            return Collections.singleton(parent);
        }
        else
        {
            return null;
        }
    }
    
    @Override
    protected boolean ignoreOnHasAnyPermission()
    {
        return true;
    }
    
    @Override
    protected Set< ? extends Object> _convertWorkspaceToRootRightContexts(Set<Object> workspacesContexts)
    {
        if (workspacesContexts.contains("/cms"))
        {
            return Set.of(_queryDAO.getQueriesRootNode());
        }
        return null;
    }
    
    @Override
    protected I18nizableText getObjectLabelForExplanation(Object object) throws RightsException
    {
        if (object instanceof Query)
        {
            Map<String, I18nizableTextParameter> params = Map.of("title", getObjectLabel(object));
            return new I18nizableText("plugin.queries-directory", "PLUGINS_QUERIESDIRECTORY_QUERY_ACCESS_CONTROLLER_QUERY_CONTEXT_LABEL", params);
        }
        else if (object instanceof QueryContainer)
        {
            if (_queryDAO.getQueriesRootNode().equals(object))
            {
                return new I18nizableText("plugin.queries-directory", "PLUGINS_QUERIESDIRECTORY_QUERY_ACCESS_CONTROLLER_ROOT_CONTAINER_CONTEXT_EXPLANATION_LABEL");
            }
            else
            {
                Map<String, I18nizableTextParameter> params = Map.of("name", getObjectLabel(object));
                return new I18nizableText("plugin.queries-directory", "PLUGINS_QUERIESDIRECTORY_QUERY_ACCESS_CONTROLLER_QUERY_CONTAINER_CONTEXT_LABEL", params);
            }
        }
        throw new RightsException("Unsupported object " + object.toString());
    }
    
    public I18nizableText getObjectLabel(Object object)
    {
        if (object instanceof Query query)
        {
            return new I18nizableText(getQueryContainerParentPathLabel(query.getParent()) + query.getTitle());
        }
        else if (object instanceof QueryContainer container)
        {
            if (_queryDAO.getQueriesRootNode().equals(object))
            {
                return new I18nizableText("plugin.queries-directory", "PLUGINS_QUERIESDIRECTORY_QUERY_ACCESS_CONTROLLER_ROOT_CONTAINER_CONTEXT_LABEL");
            }
            else
            {
                return new I18nizableText(getQueryContainerParentPathLabel(container.getParent()) + container.getName());
            }
        }
        throw new RightsException("Unsupported object " + object.toString());
    }
    
    public I18nizableText getObjectCategory(Object object)
    {
        return QUERIES_DIRECTORY_CONTEXT_CATEGORY;
    }
    
    public int getObjectPriority(Object object)
    {
        if (_queryDAO.getQueriesRootNode().equals(object))
        {
            return 10;
        }
        return super.getObjectPriority(object);
    }
    
    /**
     * Get a label representing the path of the container.
     * 
     * The root is not included in the path
     * @param container the container
     * @return the label
     */
    public static String getQueryContainerParentPathLabel(QueryContainer container)
    {
        if (container.getParent() instanceof QueryContainer parent)
        {
            return getQueryContainerParentPathLabel(parent) + container.getName() + " > ";
        }
        else
        {
            return "";
        }
    }
}
