/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.queriesdirectory.accesscontroller;

import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.collections.MapUtils;

import org.ametys.core.group.GroupIdentity;
import org.ametys.core.right.AccessController;
import org.ametys.core.right.AccessExplanation;
import org.ametys.core.right.RightsException;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.core.impl.right.AbstractRightBasedAccessController;
import org.ametys.plugins.queriesdirectory.Query;
import org.ametys.plugins.queriesdirectory.QueryDAO;
import org.ametys.plugins.queriesdirectory.QueryFactory;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.query.QueryHelper;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.UserExpression;
import org.ametys.runtime.i18n.I18nizableText;

/**
 * {@link AccessController} to allow read access and handle for author of a query
 *
 */
public class QueryAuthorAccessController extends AbstractRightBasedAccessController implements Serviceable
{
    private static final List<String> __CREATOR_RIGHTS = List.of(QueryDAO.QUERY_HANDLE_RIGHT_ID);
    /** The ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
    }
    
    public boolean supports(Object object)
    {
        return object instanceof Query;
    }
    
    public AccessResult getPermission(UserIdentity user, Set<GroupIdentity> userGroups, String rightId, Object object)
    {
        if (((Query) object).getAuthor().equals(user))
        {
            return __CREATOR_RIGHTS.contains(rightId) ? AccessResult.USER_ALLOWED : AccessResult.UNKNOWN;
        }
        
        return AccessResult.UNKNOWN;
    }

    public AccessResult getReadAccessPermission(UserIdentity user, Set<GroupIdentity> userGroups, Object object)
    {
        return ((Query) object).getAuthor().equals(user) ? AccessResult.USER_ALLOWED : AccessResult.UNKNOWN;
    }

    /**
     * If creator, access to a list of rights
     */
    public Map<String, AccessResult> getPermissionByRight(UserIdentity user, Set<GroupIdentity> userGroups, Object object)
    {
        Map<String, AccessResult> permissionByRight = new HashMap<>();
        
        if (((Query) object).getAuthor().equals(user))
        {
            for (String rightId : __CREATOR_RIGHTS)
            {
                permissionByRight.put(rightId, AccessResult.USER_ALLOWED);
            }
        }
        
        return permissionByRight;
    }

    public AccessResult getPermissionForAnonymous(String rightId, Object object)
    {
        return AccessResult.UNKNOWN;
    }

    public AccessResult getReadAccessPermissionForAnonymous(Object object)
    {
        return AccessResult.UNKNOWN;
    }

    public AccessResult getPermissionForAnyConnectedUser(String rightId, Object object)
    {
        return AccessResult.UNKNOWN;
    }

    public AccessResult getReadAccessPermissionForAnyConnectedUser(Object object)
    {
        return AccessResult.UNKNOWN;
    }

    /**
     * If right requested is in the list, the creator is added the list of USER_ALLOWED
     */
    public Map<UserIdentity, AccessResult> getPermissionByUser(String rightId, Object object)
    {
        Map<UserIdentity, AccessResult> permissionByUser = new HashMap<>();
        
        if (__CREATOR_RIGHTS.contains(rightId))
        {
            permissionByUser.put(((Query) object).getAuthor(), AccessResult.USER_ALLOWED);
        }
        return permissionByUser;
    }

    public Map<UserIdentity, AccessResult> getReadAccessPermissionByUser(Object object)
    {
        return MapUtils.EMPTY_MAP;
    }

    public Map<GroupIdentity, AccessResult> getPermissionByGroup(String rightId, Object object)
    {
        return MapUtils.EMPTY_MAP;
    }

    public Map<GroupIdentity, AccessResult> getReadAccessPermissionByGroup(Object object)
    {
        return MapUtils.EMPTY_MAP;
    }

    public boolean hasUserAnyPermissionOnWorkspace(Set<Object> workspacesContexts, UserIdentity user, Set<GroupIdentity> userGroups, String rightId)
    {
        return false;
    }

    public boolean hasUserAnyReadAccessPermissionOnWorkspace(Set<Object> workspacesContexts, UserIdentity user, Set<GroupIdentity> userGroups)
    {
        return false;
    }

    public boolean hasAnonymousAnyPermissionOnWorkspace(Set<Object> workspacesContexts, String rightId)
    {
        return false;
    }

    public boolean hasAnonymousAnyReadAccessPermissionOnWorkspace(Set<Object> workspacesContexts)
    {
        return false;
    }

    public boolean hasAnyConnectedUserAnyPermissionOnWorkspace(Set<Object> workspacesContexts, String rightId)
    {
        return false;
    }

    public boolean hasAnyConnectedUserAnyReadAccessPermissionOnWorkspace(Set<Object> workspacesContexts)
    {
        return false;
    }
    
    @Override
    protected AccessExplanation _getAccessExplanation(AccessResult result, Object object, UserIdentity user, Set<GroupIdentity> groups, String rightId)
    {
        switch (result)
        {
            case USER_ALLOWED:
            case UNKNOWN:
                return new AccessExplanation(
                        getId(),
                        result,
                        new I18nizableText("plugin.queries-directory", "PLUGINS_QUERIESDIRECTORY_QUERY_AUTHOR_ACCESS_CONTROLLER_" + result.name() + "_EXPLANATION",
                                Map.of("title", getObjectLabel(object)))
                        );
            default:
                return AccessController.getDefaultAccessExplanation(getId(), result);
        }
    }
    
    @Override
    protected Iterable< ? extends Object> getHandledObjects(UserIdentity identity, Set<GroupIdentity> groups, Set<Object> workspacesContexts)
    {
        if (workspacesContexts.contains("/cms"))
        {
            String xPathQuery = QueryHelper.getXPathQuery(null, QueryFactory.QUERY_NODETYPE, new UserExpression(Query.AUTHOR, Operator.EQ, identity));
            return _resolver.query(xPathQuery);
        }
        return List.of();
    }
    
    @Override
    protected Collection<String> getHandledRights()
    {
        return __CREATOR_RIGHTS;
    }
    
    @Override
    public I18nizableText getObjectLabel(Object object)
    {
        if (object instanceof Query query)
        {
            return new I18nizableText(QueryAccessController.getQueryContainerParentPathLabel(query.getParent()) + query.getTitle());
        }
        throw new RightsException("Unsupported context: " + object.toString());
    }
    
    @Override
    public I18nizableText getObjectCategory(Object object)
    {
        return QueryAccessController.QUERIES_DIRECTORY_CONTEXT_CATEGORY;
    }
}
