/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/* Imports */
const StringUtils = Java.type("org.apache.commons.lang3.StringUtils");
const ArrayList = Java.type("java.util.ArrayList");

/* replacement regex building */
let stringsToReplace = new ArrayList();
stringsToReplace.add("metadata-");
stringsToReplace.add("property-");
let replacementRegExp = new RegExp(StringUtils.join(stringsToReplace, "|"), "gi");
let replacementText = "reference-";

logger.info("Start migration of queries' criteria identifiers");

__migrateQueriesCriteria(Repository.session);
if (Repository.liveSession)
{
    __migrateQueriesCriteria(Repository.liveSession);
}

function __migrateQueriesCriteria(session)
{
    let migratedQueries = 0;
    try
    {
        Repository.query("//element(*, ametys:query)", true, session).forEach(function(query)
        {
            let hasChanges = false;
            let queryContent = query.getContent();
            
            let containsSearchedString = false;
            for (let stringToReplace of stringsToReplace)
            {
                containsSearchedString = StringUtils.containsIgnoreCase(queryContent, stringToReplace) || containsSearchedString;
            }
            
            if (containsSearchedString)
            {
                let migratedQueryContent = queryContent.replace(replacementRegExp, replacementText);
                query.setContent(migratedQueryContent);
                hasChanges = true;
            }
        
            if (hasChanges)
            {
                query.saveChanges();
                migratedQueries ++;
            }
        })
    }
    catch (e)
    {
        logger.error("[" + session.getWorkspace().getName() + "] An error has occured during query migration for query '" + query.getId() , e);
        throw e;
    } 
    finally 
    {
        logger.info("[" + session.getWorkspace().getName() + "] " + migratedQueries + " handled queries");
    }
}
