/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.serverdirectory;

import java.util.Base64;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.acting.ServiceableAction;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.Response;
import org.apache.cocoon.environment.SourceResolver;
import org.apache.commons.lang3.StringUtils;
import org.apache.excalibur.source.Source;

import org.ametys.core.right.RightManager;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.util.URIUtils;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.data.holder.ModelAwareDataHolder;
import org.ametys.runtime.authentication.AccessDeniedException;
import org.ametys.runtime.authentication.AuthorizationRequiredException;
import org.ametys.web.repository.page.SitemapElement;
import org.ametys.web.repository.page.ZoneItem;

/**
 *  Check if we are authorized to access to the path
 */
public class CheckPathAccessAction extends ServiceableAction
{
    private org.apache.excalibur.source.SourceResolver _srcResolver;
    private AmetysObjectResolver _resolver;
    private CurrentUserProvider _currentUserProvider;
    private RightManager _rightManager;

    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _srcResolver = (org.apache.excalibur.source.SourceResolver) smanager.lookup(org.apache.excalibur.source.SourceResolver.ROLE);
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
        _currentUserProvider = (CurrentUserProvider) smanager.lookup(CurrentUserProvider.ROLE);
        _rightManager = (RightManager) smanager.lookup(RightManager.ROLE);
    }
    
    public Map act(Redirector redirector, SourceResolver resolver, Map objectModel, String source, Parameters parameters) throws Exception
    {
        Map<String, Object> result = new HashMap<>();

        Request request = ObjectModelHelper.getRequest(objectModel);        
        String path = request.getParameter("path");
        String name = request.getParameter("name");

        if (StringUtils.isBlank(path))
        {
            throw new IllegalArgumentException("Missing server directory's path");
        }
        if (StringUtils.isBlank(name))
        {
            throw new IllegalArgumentException("Missing server file's name");
        }

        String decodedPath = ServerDirectoryHelper.normalize(URIUtils.decode(path));
        
        String zoneItemEncoded = parameters.getParameter("zoneItem");
        String zoneItemId = new String(Base64.getUrlDecoder().decode(zoneItemEncoded.getBytes("UTF-8")));
        ZoneItem zoneItem = (ZoneItem) _resolver.resolveById(zoneItemId);
        
        // Check page to page
        _checkPageAccess(zoneItem);
        
        ModelAwareDataHolder serviceParameters = zoneItem.getServiceParameters();
        boolean enableDynamicPaths = serviceParameters.getValue("enableDynamicPaths", false, false);
        
        String folder = ServerDirectoryHelper.normalize(serviceParameters.getValue("folder"));
        if (enableDynamicPaths)
        {
            String site = (String) request.getAttribute("site");
            String language = (String) request.getAttribute("sitemapLanguage");
            
            folder = ServerDirectoryHelper.evaluateDynamicPath(folder, site, language, _currentUserProvider.getUser());
            
            if (!folder.startsWith("file:/"))
            {
                folder = "file:/" + folder;
            }
        }
        
        Set<Source> rootSources = ServerDirectoryHelper.getRootServerSources(_srcResolver);
        if (!ServerDirectoryHelper.isValidPath(decodedPath, rootSources))
        {
            throw new AccessDeniedException("You are not allowed to access to server directory file " + decodedPath);
        }
        
        if (!decodedPath.startsWith(folder))
        {
            throw new IllegalStateException("The server directory file '" + decodedPath + "' is not part of the current service : " + folder);
        }
        
        result.put("path", path);
        
        String decodedName = ServerDirectoryHelper.normalize(URIUtils.decode(name));
        Response response = ObjectModelHelper.getResponse(objectModel);
        _setHeader(decodedName, response);
        
        return result;
    }
    
    private void _checkPageAccess(ZoneItem zoneItem) throws AuthorizationRequiredException, AmetysRepositoryException, AccessDeniedException
    {
        SitemapElement sitemapElement = zoneItem.getZone().getSitemapElement();
        
        if (!_rightManager.hasAnonymousReadAccess(sitemapElement))
        {
            UserIdentity user = _currentUserProvider.getUser();
            if (user == null)
            {
                throw new AuthorizationRequiredException(null);
            }
            else if (!_rightManager.hasReadAccess(user, sitemapElement))
            {
                throw new AccessDeniedException("Access to page " + sitemapElement.getSiteName() + "/" + sitemapElement.getSitemapName() + "/" + sitemapElement.getPathInSitemap() + " is not allowed for user " + user);
            }
        }
    }
    
    /**
     * Set Content-Disposition header at attachement, with the file name
     * @param name file name to encode
     * @param response request response where the header will be set
     */
    protected void _setHeader (String name, Response response)
    {
        String encodedName = URIUtils.encodeHeader(name);
        response.setHeader("Content-Disposition", "attachment; filename=\"" + encodedName + "\";filename*=UTF-8''" + encodedName);
    }
}
