/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.serverdirectory;

import java.io.IOException;
import java.net.URISyntaxException;
import java.util.Base64;
import java.util.Collection;
import java.util.Set;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.generation.ServiceableGenerator;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.excalibur.source.Source;
import org.apache.excalibur.source.SourceException;
import org.apache.excalibur.source.SourceResolver;
import org.apache.excalibur.source.TraversableSource;
import org.xml.sax.SAXException;

import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.util.URIUtils;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.data.holder.ModelAwareDataHolder;
import org.ametys.runtime.authentication.AccessDeniedException;
import org.ametys.runtime.authentication.AuthorizationRequiredException;
import org.ametys.web.repository.page.ZoneItem;

/**
 * Generate all subDirectory from a directory
 */
public class ServerDirectoryGenerator extends ServiceableGenerator
{
    /** Excalibur's source resolver */
    protected SourceResolver _sourceResolver;
    
    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;

    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        _sourceResolver = (SourceResolver) smanager.lookup(SourceResolver.ROLE);
        _currentUserProvider = (CurrentUserProvider) smanager.lookup(CurrentUserProvider.ROLE);
    }
    
    public void generate() throws IOException, SAXException, ProcessingException
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        
        ZoneItem zoneItem = (ZoneItem) request.getAttribute(ZoneItem.class.getName());
        ModelAwareDataHolder serviceParameters = zoneItem.getServiceParameters();
        String folder = ServerDirectoryHelper.normalize(serviceParameters.getValue("folder", false, ""));
        
        try
        {
            contentHandler.startDocument();
            
            if (serviceParameters.getValue("enableDynamicPaths", false, false))
            {
                folder = ServerDirectoryHelper.evaluateDynamicPath(folder, (String) request.getAttribute("site"), (String) request.getAttribute("sitemapLanguage"), _currentUserProvider.getUser());
            }
            
            Set<Source> rootSources = ServerDirectoryHelper.getRootServerSources(_sourceResolver);
            
            if (!ServerDirectoryHelper.isValidPath(folder, rootSources))
            {
                throw new IllegalStateException("The server directory '" + folder + "' is not an authorized directory");
            }
            
            Source sourceFolder = null;
            try
            {
                sourceFolder = _sourceResolver.resolveURI(folder);
                
                if (!sourceFolder.exists())
                {
                    throw new IllegalArgumentException("The server directory '" + folder + "' does not exist");
                }
                
                String zoneItemEncoded = new String(Base64.getUrlEncoder().encode(zoneItem.getId().getBytes("UTF-8")));
                Long depth = serviceParameters.getValue("depth");
                saxCollection(sourceFolder, zoneItemEncoded, depth == null || depth.longValue() != depth.intValue() ? Integer.MAX_VALUE : depth.intValue());
            }
            catch (SourceException | AmetysRepositoryException | URISyntaxException e)
            {
                throw new IllegalArgumentException("Cannot enumerate subdirectories for server location: <" + folder + ">", e);
            }
            finally
            {
                _sourceResolver.release(sourceFolder);
            }
            
            contentHandler.endDocument();
        }
        catch (AccessDeniedException | AuthorizationRequiredException e)
        {
            throw new ProcessingException("No user is connected or the user has no required permissions to access to the server directory " + folder);
        }
    }
    
    /**
     * SAX a {@link Source}
     * @param sourceFolder The source to sax
     * @param zoneItemId The zone item id
     * @param depth remaining max depth
     * @throws SAXException If an error occurred while saxing
     * @throws SourceException If an error occurred while we get child from the source
     * @throws URISyntaxException if an error occurred 
     */
    protected void saxCollection (Source sourceFolder, String zoneItemId, Integer depth) throws SAXException, SourceException, URISyntaxException
    {
        if (sourceFolder instanceof TraversableSource)
        {            
            TraversableSource tSource = (TraversableSource) sourceFolder;
            
            AttributesImpl childAtts = new AttributesImpl();
            childAtts.addCDATAAttribute("id", tSource.getURI());
            childAtts.addCDATAAttribute("name", tSource.getName());
            childAtts.addCDATAAttribute("encodedName", URIUtils.encodePathSegment(tSource.getName()));
            childAtts.addCDATAAttribute("mimetype", tSource.getMimeType());
            childAtts.addCDATAAttribute("lastModified", String.valueOf(tSource.getLastModified()));
            childAtts.addCDATAAttribute("size", String.valueOf(tSource.getContentLength()));
            childAtts.addCDATAAttribute("path", URIUtils.encodePath(tSource.getURI()));
            
            if (tSource.isCollection())
            {
                childAtts.addCDATAAttribute("type", "collection");
                XMLUtils.startElement(contentHandler, "Node", childAtts);

                if (depth > 0)
                {
                    Collection<Source> childrenSources = tSource.getChildren(); 
                    for (Source childSource : childrenSources)
                    {
                        saxCollection(childSource, zoneItemId, depth - 1);
                    }
                }
                XMLUtils.endElement(contentHandler, "Node");
            }
            else
            {
                childAtts.addCDATAAttribute("type", "resource");
                childAtts.addCDATAAttribute("zoneItem", zoneItemId);
                XMLUtils.createElement(contentHandler, "Node", childAtts);
            }
        }
    }
}
