/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.serverdirectory;

import java.util.Arrays;
import java.util.Comparator;
import java.util.HashSet;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.cocoon.ProcessingException;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.excalibur.source.Source;
import org.apache.excalibur.source.SourceResolver;

import org.ametys.core.user.UserIdentity;
import org.ametys.runtime.authentication.AccessDeniedException;
import org.ametys.runtime.authentication.AuthorizationRequiredException;
import org.ametys.runtime.config.Config;

/**
 * Helper gathering utility methods for server directories
 */
public final class ServerDirectoryHelper
{
    // Pattern for the dynamic path variables : matches "${g1}" and "${g1[g2=g3]}", with g1, g2 and g3 the capturing groups
    private static final Pattern DYNAMIC_PATH_PATTERN = Pattern.compile("\\$\\{([^}\\[]+)(?:\\[([^=\\[\\]\\}\\{]+)=([^=\\[\\]\\}\\{]+)\\])?\\}");
    
    private ServerDirectoryHelper()
    {
        // empty constructor
    }
    
    /**
     * Get the sources corresponding to the roots of the defined server directories
     * @param sourceResolver the source resolver
     * @return the sources of the server directories' roots
     * @throws ProcessingException if the sever directory's location wasn't found
     */
    public static Set<Source> getRootServerSources(SourceResolver sourceResolver) throws ProcessingException
    {
        Set<String> locations = ServerDirectoryHelper.getRootServerDirectoryPaths();
        
        Set<Source> sources = new HashSet<>();
        
        for (String location : locations)
        {
            Source src = null;
            try
            {
                src = sourceResolver.resolveURI(location.trim(), "file://", null);
                if (src.exists())
                {
                    sources.add(src);
                }
            }
            catch (Exception e)
            {
                throw new ProcessingException("Unable to retrieve server directory to location: <" + location + ">", e);
            }
            finally
            {
                sourceResolver.release(src);
            }
        }
        
        return sources;
    }
    
    /**
     * Get the paths of the server directories' roots
     * @return the paths of the server directories' roots
     */
    public static Set<String> getRootServerDirectoryPaths ()
    {
        String authorizedDirectories = Config.getInstance().getValue("org.ametys.plugins.server.directory.authorized");
        String[] locations = StringUtils.split(authorizedDirectories, "\n");     
        
        Comparator<String> c = new Comparator<>() 
        {
            @Override
            public int compare(String s1, String s2) 
            {
                
                int l1 = s1.length();
                int l2 = s2.length();
                
                return ((Integer) l1).compareTo(l2);
            }
        };
        Arrays.sort(locations, c);
        
        Set<String> distinctLocations = new HashSet<>();
        for (String location : locations)
        {
            boolean found = false;
            for (String distinctLocation : distinctLocations)
            {
                if (location.startsWith(distinctLocation))
                {
                    found = true;
                    break;
                }
            }
            
            if (!found)
            {
                distinctLocations.add(location.trim());
            }
        }
        
        return distinctLocations;
    }
    
    /**
     * Normalizes a path, removing double and single dot path steps and replace '\' by '/'
     * @param path The path
     * @return The normalized path
     */
    public static String normalize (String path)
    {
        String normalizedPath = FilenameUtils.normalize(path);
        return normalizedPath.replace("\\", "/");
    }
    
    /**
     * Check if the given path is a valid path for a root of a server directory
     * @param path the path to check
     * @param rootSources the sources corresponding to the server directories' roots
     * @return true if the path is valid, false otherwise
     */
    public static boolean isValidPath(String path, Set<Source> rootSources)
    {
        String normalizedPath = normalize(path);
        for (Source rootSource : rootSources)
        {
            if (normalizedPath.startsWith(rootSource.getURI()))
            {
                return true;
            }
        }
        return false;
    }
    
    /**
     * Evaluate a dynamic path with contextual variables
     * @param path The dynamic path
     * @param siteName The current site name
     * @param language The current site language
     * @param currentUser The current user
     * @return The path resolved
     * @throws AuthorizationRequiredException If the current user is null and the path required the login
     * @throws AccessDeniedException If the connected user does not belong to required population
     * @throws IllegalArgumentException If a dynamic variable can not be evaluated
     */
    public static String evaluateDynamicPath(String path, String siteName, String language, UserIdentity currentUser) throws AuthorizationRequiredException, AccessDeniedException, IllegalArgumentException
    {
        Matcher matcher = DYNAMIC_PATH_PATTERN.matcher(path);
        
        StringBuffer newPath = new StringBuffer();
        while (matcher.find())
        {
            String variable = matcher.group(1);
            String param = matcher.group(2);
            String paramValue = matcher.group(3);
            
            if ("login".equals(variable))
            {
                if (currentUser == null)
                {
                    throw new AuthorizationRequiredException(null);
                }
                if (param != null && paramValue != null)
                {
                    if ("population".equals(param))
                    {
                        String[] populations = paramValue.split(",");
                        if (!ArrayUtils.contains(populations, currentUser.getPopulationId()))
                        {
                            throw new AccessDeniedException("The user " + currentUser + " is not authorized to access file " + path);
                        }
                    }
                    else
                    {
                        // unknown parameter
                        throw new IllegalArgumentException("Unable to evaluate the current site");
                    }
                }
                
                matcher.appendReplacement(newPath, currentUser.getLogin());
            }
            else if ("population".equals(variable))
            {
                if (currentUser == null)
                {
                    throw new AuthorizationRequiredException(null);
                }
                
                matcher.appendReplacement(newPath, currentUser.getPopulationId());
            }
            else if ("site".equals(variable))
            {
                if (siteName == null)
                {
                    throw new IllegalArgumentException("Unable to evaluate the current site for service directory path " + path);
                }
                
                matcher.appendReplacement(newPath, siteName);
            }
            else if ("lang".equals(variable))
            {
                if (language == null)
                {
                    throw new IllegalArgumentException("Unable to evaluate the current language for service directory path " + path);
                }
                
                matcher.appendReplacement(newPath, language);
            }
            else
            {
                matcher.appendReplacement(newPath, "$0");
            }
        }
        
        matcher.appendTail(newPath);
        
        return newPath.toString();
    }
}
