/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.serverdirectory;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.acting.ServiceableAction;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.commons.lang3.StringUtils;
import org.apache.excalibur.source.Source;
import org.apache.excalibur.source.SourceResolver;
import org.apache.excalibur.source.TraversableSource;

import org.ametys.core.cocoon.JSonReader;

/**
 * Generate a subDirectories list from a query and a list of root paths from the configuration
 */
public class ServerDirectoryPathAction extends ServiceableAction
{
    /** Excalibur's source resolver */
    private SourceResolver _sourceResolver;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        _sourceResolver = (SourceResolver) smanager.lookup(SourceResolver.ROLE);
    }
    
    public Map act(Redirector redirector, org.apache.cocoon.environment.SourceResolver resolver, Map objectModel, String source, Parameters parameters) throws Exception
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        Map<String, Object> result = new HashMap<> ();
        
        List<Map<String, Object>> items = new ArrayList<> ();
        boolean enableDynamicPaths = "true".equals(request.getParameter("enableDynamicPaths"));

        Set<Source> sources = new HashSet<>();
        
        String queryPath = ServerDirectoryHelper.normalize(parameters.getParameter("query", ""));
        
        Set<Source> rootSources = ServerDirectoryHelper.getRootServerSources(_sourceResolver);
        for (Source rootSource : rootSources)
        {
            if (StringUtils.isEmpty(queryPath))
            {
                sources.add(rootSource);
            }
            else
            {
                String rootURI = rootSource.getURI();
                String rootPath = StringUtils.substringAfter(rootURI, "file:/");
                
                if (queryPath.startsWith(rootPath) || queryPath.startsWith(rootURI))
                {
                    if (!enableDynamicPaths)
                    {
                        sources.addAll(_getValidServerDirectories(queryPath));
                    }
                }
                else if (rootPath.startsWith(queryPath) || rootURI.startsWith(queryPath))
                {
                    sources.add(rootSource);
                }
            }
        }
        
        for (Source pathResult : sources)
        {
            Map<String, Object> item = new HashMap<> ();
            item.put("displayText", StringUtils.substringAfter(pathResult.getURI(), pathResult.getScheme() + ":/"));
            item.put("path", pathResult.getURI());
            
            items.add(item);
        }
        
        result.put("items", items);
        
        request.setAttribute(JSonReader.OBJECT_TO_READ, result);
        return EMPTY_MAP;
    }
    
    /**
     * Look for valid directories inside of a path.
     * @param path The path. It can be incomplete and a partial path will be used to match multiple folders.
     * @return a set of matching directories paths
     * @throws ProcessingException if an error occurs when retrieving the list of valid server directories
     */
    private Set<Source> _getValidServerDirectories(String path) throws ProcessingException
    {
        Set<Source> validSources = new HashSet<>();
        
        String folderStartsWith = "";
        String rootPath = path.replace("\\", "/");
        
        if (!rootPath.endsWith("/"))
        {
            if (rootPath.contains("/"))
            {
                folderStartsWith = rootPath.substring(rootPath.lastIndexOf("/") + 1);
                rootPath = rootPath.substring(0, rootPath.lastIndexOf("/"));
            }
            else
            {
                folderStartsWith = rootPath;
                rootPath = "";
            }
        }
        
        Source root = null;
        try
        {
            root = _sourceResolver.resolveURI(rootPath.trim(), "file://", null);
            if (root.exists() && root instanceof TraversableSource && ((TraversableSource) root).isCollection())
            {
                if (folderStartsWith.isEmpty())
                {
                    validSources.add(root);
                }
                
                TraversableSource tSource = (TraversableSource) root;
                Collection<Source> childrenSources = tSource.getChildren();
                for (Source childSource : childrenSources)
                {
                    if (childSource instanceof TraversableSource && ((TraversableSource) childSource).isCollection())
                    { 
                        String[] childPathSplit = childSource.getURI().split("/");
                        if (childPathSplit.length > 0 && childPathSplit[childPathSplit.length - 1].startsWith(folderStartsWith))
                        {
                            validSources.add(childSource);
                        }
                    }
                    
                }
            }
        }
        catch (Exception e)
        {
            throw new ProcessingException("Unable to retrieve server directory to location: <" + rootPath + ">", e);
        }
        finally
        {
            _sourceResolver.release(root);
        }
        
        return validSources;
    }

    
}
