/*
 *  Copyright 2013 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * This helper allows to choose mode before enter in edition. See #open method.
 */
Ext.define('Ametys.plugins.skincommons.helper.ChooseOpenMode', {
    singleton: true,

    /**
     * Open a dialog to choose the edition mode of the skin.
     * @param {Object} config The configuration object
     * @param {String} config.iconCls One or more space separated CSS classes to be applied to the dialog icon
     * @param {String} config.title The title for dialog box
     * @param {String} config.helpMessage The help message
     * @param {String} [config.lockDate] The lock date
     * @param {String} [config.lastSave] The last modification date
     * @param {Function} config.callback The callback function
     * @param {Function} config.callback.mode The chosen mode: 'temp', 'work' or 'prod'. Can be null if user cancelled the dialog box.
     */
    open : function (config)
    {
        this._cb = config.callback;
        
        this.delayedInitialize (config.iconCls, config.title, config.helpMessage);
        this._init(config.lockDate, config.lastSave);
        
        this._box.show();
    },

    /**
     * @private
     * Creates the dialog box if needed
     * @param {String} iconCls One or more space separated CSS classes to be applied to the dialog icon
     * @param {String} title The title of the dialog
     * @param {String} helpmessage The help message
     */
    delayedInitialize : function (iconCls, title, helpmessage)
    {
        if (!this._initialized)
        {
	        this._form = Ext.create('Ext.form.FormPanel', {
	            border: false,
	            scrollable: true,
	            defaults:
	            {
	                cls: 'ametys',
	                anchor: '95%',
	                labelAlign: 'right',
	                labelSeparator: '',
	                labelWidth: 50
	            },
	
	            items: [
	                {
	                    xtype: 'container',
	                    cls: 'text',
	                    html: helpmessage
	                },
	                {
	                    xtype: 'radio',
	                    hideLabel: true,
	                    itemCls: 'checkbox',
	                    boxLabel: "{{i18n PLUGINS_SKINCOMMONS_CHOOSEOPENMODE_TEMP}}",
	                    id: 'mode-temp',
	                    name: 'mode',
	                    inputValue: "temp",
	                    checked:true
	                },
	                {
	                    xtype: 'radio',
	                    hideLabel: true,
	                    itemCls: 'checkbox',
	                    boxLabel: "{{i18n PLUGINS_SKINCOMMONS_CHOOSEOPENMODE_WORK}}",
	                    id: 'mode-work',
	                    name: 'mode',
	                    inputValue: "work"
	                },
	                {
	                    xtype: 'radio',
	                    hideLabel: true,
	                    itemCls: 'checkbox',
	                    boxLabel: "{{i18n PLUGINS_SKINCOMMONS_CHOOSEOPENMODE_PROD}}",
	                    id: 'mode-prod',
	                    name: 'mode',
	                    inputValue: "prod"
	                 }
	            ]
	        });
	        
	        this._box = Ext.create('Ametys.window.DialogBox', {
	            title: title,
	            iconCls: iconCls,
	            
	            border: false,
	            width : 540,
	            layout: 'fit',
	            
	            items: [this._form],
	            
	            defaultFocus: 'mode-temp',
	            closeAction: 'hide',
	            
	            referenceHolder: true,
	            defaultButton: 'validate',
	            
	            buttons: [{
	                	reference: 'validate',
	                    text :"{{i18n PLUGINS_SKINCOMMONS_CHOOSEOPENMODE_OK}}",
	                    handler : this._validate,
	                    scope: this
	                }, {
	                    text :"{{i18n PLUGINS_SKINCOMMONS_CHOOSEOPENMODE_CANCEL}}",
	                    handler : this._cancel,
	                    scope: this
	                }]
	        });
        
	        this._initialized = true;
        }
        else
        {
        	this._box.setTitle(title);
            this._box.setIconCls(iconCls);
            this._form.items.get(0).update (helpmessage);
        }
    },

    /**
     * @private
     * Initialize the dialog
     * @param {String} lockDate The lock date
     * @param {String} lastSave The date of the last save 
     */
    _init : function (lockDate, lastSave)
    {
        var form = this._form.getForm();
        
        var tempFd = form.findField('mode-temp');
        if (!Ext.isEmpty(lockDate))
        {
            tempFd.setVisible(true);
            tempFd.setValue(true);
            
            tempFd.boxLabel = "{{i18n PLUGINS_SKINCOMMONS_CHOOSEOPENMODE_TEMP}}" + '<b>' + lockDate + '</b>';
            if (tempFd.rendered)
            {
                tempFd.setBoxLabel(tempFd.boxLabel);
            }
        }
        else
        {
            tempFd.setVisible(false);
            tempFd.setValue(false);
        }
        
        var workFd = form.findField('mode-work');
        if (!Ext.isEmpty(lastSave))
        {
            workFd.setVisible(true);
            workFd.setValue(lockDate == null);
            
            if (lockDate != null)
            {
                workFd.boxLabel = "{{i18n PLUGINS_SKINCOMMONS_CHOOSEOPENMODE_WORK_2}}" + '<b>' + lastSave + '</b>';
            }
            else
            {
                workFd.boxLabel = "{{i18n PLUGINS_SKINCOMMONS_CHOOSEOPENMODE_WORK}}" + '<b>' + lastSave + '</b>';
            }
            
            if (workFd.rendered)
            {
                workFd.setBoxLabel(workFd.boxLabel);
            }
        }
        else
        {
            workFd.setVisible(false);
            workFd.setValue(false);
        }
    },

    /**
     * @private
     * Function invoked when clicking on 'Ok' button
     */
    _validate : function ()
    {
        this._box.hide();
        
        if (Ext.isFunction(this._cb))
        {
            var mode = this._form.getForm().getValues().mode;
            this._cb (mode);
        }
    },

    /**
     * @private
     * Function invoked when clicking on 'Cancel' button
     */
    _cancel : function ()
    {
        this._box.hide();
        
        if (Ext.isFunction(this._cb))
        {
            this._cb (null);
        }
    }
});