/*
 *  Copyright 2013 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.skincommons;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Arrays;
import java.util.Date;
import java.util.stream.Stream;

import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathExpressionException;
import javax.xml.xpath.XPathFactory;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.component.ComponentException;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.avalon.framework.thread.ThreadSafe;
import org.apache.cocoon.Constants;
import org.apache.cocoon.i18n.BundleFactory;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.comparator.LastModifiedFileComparator;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.RandomStringUtils;
import org.xml.sax.InputSource;

import org.ametys.core.cocoon.XMLResourceBundleFactory;
import org.ametys.core.util.path.PathUtils;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.runtime.servlet.RuntimeConfig;
import org.ametys.web.cache.CacheHelper;
import org.ametys.web.cache.pageelement.PageElementCache;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.site.SiteManager;
import org.ametys.web.skin.SkinsManager;

/**
 * Helper for skin edition
 *
 */
public class SkinEditionHelper extends AbstractLogEnabled implements Component, ThreadSafe, Serviceable, Contextualizable
{
    /** The Avalon role name */
    public static final String ROLE = SkinEditionHelper.class.getName();
    
    private static final DateFormat _DATE_FORMAT = new SimpleDateFormat("yyyy-MM-dd-HHmmss");
    
    /** The cocoon context */
    protected org.apache.cocoon.environment.Context _cocoonContext;
    
    private SiteManager _siteManager;
    private PageElementCache _zoneItemCache;
    private PageElementCache _inputDataCache;
    private XMLResourceBundleFactory _i18nFactory;
    private SkinsManager _skinsManager;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        _siteManager = (SiteManager) smanager.lookup(SiteManager.ROLE);
        _zoneItemCache = (PageElementCache) smanager.lookup(PageElementCache.ROLE + "/zoneItem");
        _inputDataCache = (PageElementCache) smanager.lookup(PageElementCache.ROLE + "/inputData");
        _i18nFactory = (XMLResourceBundleFactory) smanager.lookup(BundleFactory.ROLE);
        _skinsManager = (SkinsManager) smanager.lookup(SkinsManager.ROLE);
    }
    
    @Override
    public void contextualize(Context context) throws ContextException
    {
        _cocoonContext = (org.apache.cocoon.environment.Context) context.get(Constants.CONTEXT_ENVIRONMENT_CONTEXT);
    }
    
    /**
     * Create a backup file of the current skin
     * @param skinName The skin name
     * @return The created backup directory
     * @throws IOException If an error occurred
     */
    public Path createBackupFile (String skinName) throws IOException
    {
        Path backupDir = getBackupDirectory(skinName, new Date());
        PathUtils.moveDirectoryToDirectory(getSkinDirectory(skinName), backupDir, true);
        return backupDir;
    }
    
    /**
     * Asynchronous file deletion
     * @param file The file to delete
     * @return <code>true</code> if the deletion succeeded
     * @throws IOException if an error occurs while manipulating files
     */
    public boolean deleteQuicklyDirectory(Path file) throws IOException
    {
        Path toDelete = file.getParent().resolve(file.getFileName() + "_todelete_" + RandomStringUtils.secure().next(10, false, true));
        
        try
        {
            // Move file
            Files.move(file, toDelete);
            
            // Then delete it in asynchronous mode
            Thread th = new Thread(new AsynchronousPathDeletion(toDelete));
            th.start();
            
            return true;
        }
        catch (IOException e)
        {
            return false;
        }
    }
    
    /**
     * Remove the old backup
     * @param skinName The skin name
     * @param keepMax The max number of backup to keep
     * @throws IOException if an error occurs while manipulating files
     */
    public void deleteOldBackup (String skinName, int keepMax) throws IOException
    {
        // Remove old backup (keep only the 5 last backup)
        File rootBackupDir = getRootBackupDirectory(skinName);
        File[] allBackup = rootBackupDir.listFiles();
        Arrays.sort(allBackup, LastModifiedFileComparator.LASTMODIFIED_REVERSE);
        
        int index = 0;
        for (File f : allBackup)
        {
            if (index > keepMax - 1)
            {
                deleteQuicklyDirectory(f.toPath());
            }
            index++;
        }
    }
    
    /**
     * Invalidate all caches relative to the modified skin.
     * @param skinName the modified skin name.
     * @throws Exception if an error occurs.
     */
    public void invalidateCaches(String skinName) throws Exception
    {
        Exception ex = null;
        
        // Invalidate the caches for sites with the skin.
        for (Site site : _siteManager.getSites())
        {
            if (skinName.equals(_getSkinId(site)))
            {
                try
                {
                    String siteName = site.getName();
                    
                    // Invalidate static cache.
                    CacheHelper.invalidateCache(site, getLogger());
                    
                    // Invalidate the page elements caches.
                    _zoneItemCache.clear(null, siteName);
                    _inputDataCache.clear(null, siteName);
                }
                catch (Exception e)
                {
                    getLogger().error("Error clearing the cache for site " + site.toString());
                    ex = e;
                }
            }
        }
        
        // If an exception was thrown, re-throw it.
        if (ex != null)
        {
            throw ex;
        }
    }
    
    private String _getSkinId(Site site)
    {
        return site.getSkinId();
    }
    
    /**
     * Invalidate all catalogs of the temporary skin
     * @param skinName The site name
     */
    public void invalidateTempSkinCatalogues(String skinName)
    {
        _invalidateSkinCatalogues(getTempDirectory(skinName), skinName, "ametys-home://skins/temp/" + skinName + "/i18n");
    }
    
    /**
     * Invalidate a catalog of the temporary skin for a given language
     * @param skinName The site name
     * @param lang The language
     */
    public void invalidateTempSkinCatalogue(String skinName, String lang)
    {
        _invalidateSkinCatalogue(getTempDirectory(skinName), skinName, "ametys-home://skins/temp/" + skinName + "/i18n", lang);
    }
    
    /**
     * Invalidate all catalogs of the skin
     * @param skinName The skin name
     */
    public void invalidateSkinCatalogues(String skinName)
    {
        _invalidateSkinCatalogues(getSkinDirectory(skinName), skinName, "skin-raw:" + skinName + "://i18n");
    }
    
    /**
     * Invalidate a catalog of the skin for a given language
     * @param skinName The site name
     * @param lang The language
     */
    public void invalidateSkinCatalogue(String skinName, String lang)
    {
        _invalidateSkinCatalogue(getSkinDirectory(skinName), skinName, "skin-raw:" + skinName + "://i18n", lang);
    }
    
    /**
     * Invalidate all catalogs of the skin
     * @param skinDir The skin directory
     * @param skinName The skin name
     * @param catalogLocation the catalog location
     */
    private void _invalidateSkinCatalogues(Path skinDir, String skinName, String catalogLocation)
    {
        Path i18nDir = skinDir.resolve("i18n");
        if (Files.exists(i18nDir))
        {
            try (Stream<Path> s = Files.list(i18nDir))
            {
                s.filter(Files::isRegularFile)
                    .forEach(i18nFile ->
                    {
                        String filename = i18nFile.getFileName().toString();
                        if (filename.equals("messages.xml"))
                        {
                            _invalidateSkinCatalogue(skinDir, skinName, catalogLocation, StringUtils.EMPTY);
                        }
                        else if (filename.startsWith("messages_"))
                        {
                            String lang = filename.substring("messages_".length(), "messages_".length() + 2);
                            _invalidateSkinCatalogue(skinDir, skinName, catalogLocation, lang);
                        }
                    });
            }
            catch (IOException e)
            {
                throw new RuntimeException("Cannot invalidate skin catalogs for skin " + skinName + " and location " + catalogLocation, e);
            }
        }
    }
    
    /**
     * Invalidate catalog of the skin
     * @param skinDir The skin directory
     * @param skinName The site name
     * @param catalogLocation the catalog location
     * @param lang The language of catalog. Can be empty.
     */
    private void _invalidateSkinCatalogue(Path skinDir, String skinName, String catalogLocation, String lang)
    {
        try
        {
            String localName = lang;
            if (StringUtils.isNotEmpty(lang))
            {
                Path f = skinDir.resolve("i18n/messages_" + lang + ".xml");
                if (!Files.exists(f))
                {
                    localName = "";
                }
            }
            
            _i18nFactory.invalidateCatalogue(catalogLocation, "messages", localName);
        }
        catch (ComponentException e)
        {
            getLogger().warn("Unable to invalidate i18n catalog for skin " + skinName + " and location " + catalogLocation , e);
        }
    }
    
    /**
     * Get the temp directory of skin
     * @param skinName The skin name
     * @return The temp directory
     */
    public Path getTempDirectory(String skinName)
    {
        return RuntimeConfig.getInstance().getAmetysHome().toPath().resolve("skins/temp/" + skinName);
    }
    
    /**
     * Get the work directory of skin
     * @param skinName The skin name
     * @return The work directory
     */
    public Path getWorkDirectory(String skinName)
    {
        return RuntimeConfig.getInstance().getAmetysHome().toPath().resolve("skins/work/" + skinName);
    }
    
    /**
     * Get the backup directory of skin
     * @param skinName The skin name
     * @param date The date
     * @return The backup directory
     */
    public Path getBackupDirectory (String skinName, Date date)
    {
        String dateStr = _DATE_FORMAT.format(date);
        return RuntimeConfig.getInstance().getAmetysHome().toPath().resolve("skins/backup/" + skinName + "/" + dateStr);
    }
    
    /**
     * Get the root backup directory of skin
     * @param skinName The skin name
     * @return The root backup directory
     */
    public File getRootBackupDirectory (String skinName)
    {
        return FileUtils.getFile(RuntimeConfig.getInstance().getAmetysHome(), "skins", "backup", skinName);
    }
    
    /**
     * Get the temp directory of skin
     * @param skinName The skin name
     * @return The temp directory URI
     */
    public String getTempDirectoryURI (String skinName)
    {
        return "ametys-home://skins/temp/" + skinName;
    }
    
    /**
     * Get the work directory of skin
     * @param skinName The skin name
     * @return The work directory URI
     */
    public String getWorkDirectoryURI (String skinName)
    {
        return "ametys-home://skins/work/" + skinName;
    }
    
    /**
     * Get the backup directory of skin
     * @param skinName The skin name
     * @param date The date
     * @return The backup directory URI
     */
    public String getBackupDirectoryURI (String skinName, Date date)
    {
        return "ametys-home://skins/backup/" + skinName + "/" + _DATE_FORMAT.format(date);
    }
    
    /**
     * Get the root backup directory of skin
     * @param skinName The skin name
     * @return The root backup directory URI
     */
    public String getRootBackupDirectoryURI (String skinName)
    {
        return "ametys-home://skins/backup/" + skinName;
    }
    
    /**
     * Get the skin directory of skin
     * @param skinName The skin name
     * @return The skin directory
     */
    public Path getSkinDirectory (String skinName)
    {
        return _skinsManager.getSkin(skinName).getRawPath();
    }
    
    /**
     * Get the model of temporary version of skin
     * @param skinName The skin name
     * @return the model name
     */
    public String getTempModel (String skinName)
    {
        return _getModel(getTempDirectory(skinName));
    }
    
    /**
     * Get the model of working version of skin
     * @param skinName The skin name
     * @return the model name
     */
    public String getWorkModel (String skinName)
    {
        return _getModel (getWorkDirectory(skinName));
    }
    
    /**
     * Get the model of the skin
     * @param skinName skinName The skin name
     * @return The model name or <code>null</code>
     */
    public String getSkinModel (String skinName)
    {
        return _getModel(getSkinDirectory(skinName));
    }
    
    private String _getModel(Path skinDir)
    {
        Path modelFile = skinDir.resolve("model.xml");
        if (!Files.exists(modelFile))
        {
            // No model
            return null;
        }

        try (InputStream is = Files.newInputStream(modelFile))
        {
            XPath xpath = XPathFactory.newInstance().newXPath();
            return xpath.evaluate("model/@id", new InputSource(is));
        }
        catch (IOException e)
        {
            getLogger().error("Can not determine the model of the skin", e);
            return null;
        }
        catch (XPathExpressionException e)
        {
            throw new IllegalStateException("The id of model is missing", e);
        }
    }
}
