/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * {@link Ametys.file.AbstractFileExplorerTree} implementation for skin resources.
 * @private
 */
Ext.define('Ametys.plugins.skineditor.SkinResourcesTree',
{
    extend: "Ametys.file.AbstractFileExplorerTree",

    /**
     * @property {String} _skinName The name of the skin
     */
    
    constructor: function (config)
    {
        this._skinName = config.skinName;
        this.callParent(arguments);
    },

    createTreeStore: function ()
    {
        return Ext.create('Ext.data.TreeStore', {
            model: 'Ametys.plugins.skineditor.SkinResourcesTree.SkinResourcesNode',
            proxy:
            {
                type: 'ametys',
                plugin: 'skineditor',
                url: 'skin/resources.json',
                reader:
                {
                    type: 'json',
                    rootProperty: 'nodes'
                }
            },
            folderSort: true,
            sorters: [{ property: 'name', direction: 'ASC'}],
            root:
            {
                hasChild: true,
                editable: false,
                allowDrag: false,
                allowDrop: true,
                type: 'root',
                expanded: true,
                loaded: true // Prevent AJAX request
            }
        });
    },
    
    _onBeforeLoad: function (store, operation, eOpts)
	{
		this.callParent(arguments);
		
		operation.setParams( Ext.apply(operation.getParams(), {
			skinName: this._skinName
		}));
	},

    editNode: function (node, oldname, newname, callback)
	{		
    	Ametys.plugins.skineditor.resources.SkinResourcesActions.rename(this._skinName, node.get('path'), newname, node.get('type'), Ext.bind(this._editNodeCb, this, [node, oldname], true));
	},
	
	/**
	 * @private
	 * Callback function called after renaming 
	 * @param {Boolean} success True if the rename action succeeded
	 * @param {String} path The path of renamed file
	 * @param {String} name The name of the file
	 * @param {Ext.data.Model} node The node.
	 * @param {String} oldName The previous name
	 */
	_editNodeCb: function(success, path, name, node, oldName)
	{
		if (!success)
		{
			this.updateNodeName(node, oldName);
		}
	},

    /**
     * Set the skin name
     * @param {String} skinName the skin name
     */
    setSkin: function (skinName)
    {
        this._skinName = skinName;

        var root = this.getRootNode();
        root.set('path', '');
        root.set('text', skinName);
        root.set('name', skinName);
        root.commit();
    },

    /**
     * Get the message target configuration for given node
     * @param {Ext.data.Model} node The node.
     * @param {Boolean} busMessage True if it is for a bus message, false if it is for the drag and drop.
     */
    getMessageTargetConfiguration: function(node, busMessage)
	{
        if (busMessage && node.isRoot())
        {
            return {
                id: Ametys.message.MessageTarget.SKIN_ROOT,
                parameters: {
                    name: this._skinName,
                    path: '',
                    skinName: this._skinName
                }
            }
        }
        else if (node.get('type') == Ametys.file.AbstractFileExplorerTree.TYPE_RESOURCE)
        {
            return {
                id: Ametys.message.MessageTarget.SKIN_RESOURCE,
                parameters: {
                    name: node.get('name'),
                    path: node.get('path'),
                    skinName: node.get('skinName')
                }   
            }
        }
        else
        {
            return {
                id: Ametys.message.MessageTarget.SKIN_COLLECTION,
                parameters: {
                    name: node.get('name'),
                    path: node.get('path'),
                    skinName: node.get('skinName')
                }   
            }
        }
	},

    _getThumbnail: function(node)
    {
        return Ametys.getPluginDirectPrefix('skineditor') + "/resource/image?path=" + node.get('path') + "&skin=" + this._skinName + "&maxWidth=100&maxHeight=100";
    },
    
    getMessageTargetIdForResource: function (type)
    {
        return Ametys.message.MessageTarget.SKIN_RESOURCE;
    },
    
    getMessageTargetIdForCollection: function (type)
    {
        return Ametys.message.MessageTarget.SKIN_COLLECTION;
    },
    
    testTarget: function (target)
    {
        return target.getId() == Ametys.message.MessageTarget.SKIN_COLLECTION || target.getId() == Ametys.message.MessageTarget.SKIN_RESOURCE;
    }
});
