/*
 *  Copyright 2013 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * This UI helper provides a dialog box to choose to unlink model or not.
 * See #open method.
 * @private
 */
Ext.define('Ametys.plugins.skineditor.skin.UnlinkModel',
{
    singleton: true,

    /**
     * @property {Boolean} _initialized True if the dialog box has been initialized.
     * @private
     */
    /**
     * @property {Ext.form.Basic} _form The inner basic form.
     * @private
     */
    /**
     * @property {Ametys.window.DialogBox} _box The dialog box
     * @private
     */

    /**
     * Allow the user to upload a file from local hard drive
     * @param {String} modelName The name of the model
     * @param {String} skinName The name of the skin
     * @param {Function} callback The callback function called when the file was uploaded. Has the following parameters
     * @param {String} callback.path The path of uploaded file
     * @param {String} callback.parentPath The path of parent folder
     * @param {Boolean} callback.reload True if a reload is needed
     */
    open: function (modelName, skinName, callback)
    {
        this._cbFn = callback;

        this._delayedInitialize();
        this._box.show();

        this._initForm(modelName, skinName);
    },

    /**
     * Initialize the dialog box
     * @return {Boolean} true if the box has been initialized (or if it was already initialized).
     * @private
     */
    _delayedInitialize: function ()
    {
        if (this._initialized)
        {
            return true;
        }

        var formPanel = Ext.create('Ext.form.Panel',
        {
            border: false,
            scrollable: true,
            defaults:
            {
                cls: 'ametys',
                anchor: '95%',
                hideLabel: true,
                labelSeparator: '',
                labelWidth: 50
            },

            items: [{
                xtype: 'container',
                cls: 'text',
                html: ''
            },
            {
                xtype: 'radio',
                hideLabel: true,
                boxLabel: "{{i18n PLUGINS_SKINEDITOR_OPEN_UNLINK_MODEL}}",
                name: 'unlink',
                id: 'unlink-0',
                inputValue: "true",
                checked: true
            },
            {
                xtype: 'radio',
                hideLabel: true,
                boxLabel: "{{i18n PLUGINS_SKINEDITOR_OPEN_KEEP_MODEL}}",
                name: 'unlink',
                id: 'unlink-1',
                inputValue: "false"
            },
            {
                xtype: 'hidden',
                name: 'skinName'
            }]
        });

        this._form = formPanel.getForm();

        this._box = Ext.create('Ametys.window.DialogBox', {
            title: "{{i18n PLUGINS_SKINEDITOR_TOOL_LABEL}}",
            iconCls: 'ametysicon-html25 decorator-ametysicon-painter14',
            width: 550,

            items: formPanel,
            
            defaultFocus: 'unlink-0',
            closeAction: 'hide',
            
            referenceHolder: true,
            defaultButton: 'validate',
            
            buttons: [
            {
            	reference: 'validate',
                text: "{{i18n plugin.skincommons:PLUGINS_SKINCOMMONS_CANCELCHANGES_DIALOG_OK_BTN}}",
                handler: Ext.bind(this._ok, this)
            },
            {
                text: "{{i18n plugin.skincommons:PLUGINS_SKINCOMMONS_CANCELCHANGES_DIALOG_CANCEL_BTN}}",
                handler: Ext.bind(this._cancel, this)
            }]
        });

        this._initialized = true;
        return true;
    },

    /**
     * Initialize or reinitialize the form with the appropriate values.
     * @param {String} modelName The name of the model
     * @param {String} skinName The name of the skin
     * @private
     */
    _initForm: function (modelName, skinName)
    {
        var msg = "{{i18n PLUGINS_SKINEDITOR_OPEN_UNLINK_MODEL_HINT_1}}" + modelName + "{{i18n PLUGINS_SKINEDITOR_OPEN_UNLINK_MODEL_HINT_2}}";
        this._box.items.get(0).items.get(0).update(msg);
        this._form.findField('unlink-0').setValue(true);
        this._form.findField('skinName').setValue(skinName);
    },

    /**
     * This function is called when the 'Ok' button of the dialog box is pressed
     * send the selected value to the callback
     * @private
     */
    _ok: function ()
    {
        var values = this._form.getValues();

        if (Ext.isFunction(this._cbFn))
        {
            this._cbFn(values.unlink == "true");
        }

        this._box.hide();
    },

    /**
     * This function is called when the 'cancel' button of the dialog box is pressed
     * @private
     */
    _cancel: function ()
    {
        if (Ext.isFunction(this._cbFn))
        {
            this._cbFn(null);
        }
        
        this._box.hide();
    }

});

