/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * This is a relation handler between:
 * * source : skin resource or collection
 * * destination : skin collection
 * 
 * E.g. when you drag a resource or collection to a collection
 * @private
 */
Ext.define('Ametys.plugins.skineditor.relations.MoveResourceRelationHandler', {
	extend: 'Ametys.relation.RelationHandler',
	
	/** 
	 * @protected
	 * @property {RegExp} sourceTargetId The message target type of the source to handle
	 */
	sourceTargetId: /^(skin-resource|skin-collection)$/,
	
	/**
	 * @protected
	 * @property {RegExp} destTargetId The message target type of the target to handle
	 */
	destTargetId: /^(skin-collection)$/,

	/**
	 * Revert the #targetId test
	 * @param {Object/Ametys.message.MessageTarget} target The target to test
	 * @return {Boolean} True if the target type does not match #targetId
	 * @private
	 */
	_nonMatchingSource: function(target)
	{
		return !this.sourceTargetId.test(target.isMessageTarget ? target.getId() : target.id);
	},
	
	/**
	 * Revert the #targetId test
	 * @param {Object/Ametys.message.MessageTarget} target The target to test
	 * @return {Boolean} True if the target type does not match #targetId
	 * @private
	 */
	_nonMatchingDest: function(target)
	{
		return !this.destTargetId.test(target.isMessageTarget ? target.getId() : target.id);
	},

	supportedRelations: function(source, target)
	{
		var sourceContentTargets = Ametys.message.MessageTargetHelper.findTargets(source.targets, this.sourceTargetId, 1);
		if (sourceContentTargets.length != 1 || Ametys.message.MessageTargetHelper.findTargets(source.targets, Ext.bind(this._nonMatchingSource, this)).length > 0)
		{
			return [];
		}

		var destContentTargets = Ametys.message.MessageTargetHelper.findTargets(target.targets, this.destTargetId, 1);
		if (destContentTargets.length != 1 || Ametys.message.MessageTargetHelper.findTargets(target.targets, Ext.bind(this._nonMatchingDest, this)).length > 0)
		{
			return [];
		}

		var relations = [ 
		    Ext.create('Ametys.relation.Relation', {
    			type: Ametys.relation.Relation.MOVE,
    			label: "{{i18n PLUGINS_SKINEDITOR_RELATIONS_MOVE_RESOURCE_LABEL}}",
    			description: "{{i18n PLUGINS_SKINEDITOR_RELATIONS_MOVE_RESOURCE_DESCRIPTION}}",
    			smallIcon: null,
    			mediumIcon: null,
    			largeIcon: null
    		})
		];

		return relations;
	},

	/**
	 * Do the #link work but based only on targets array
	 * The method is called to link source to target using the given relation. 
	 * This operation can be asynchronous and will call callback at the end.
	 * In most cases this implementation will send a Ametys.message.Message to inform the UI that the operation was done.
	 * @param {Ametys.message.MessageTarget[]} sourceTargets The source point of the link operation. Targets are assumed to be ready.
	 * @param {Ametys.message.MessageTarget} target The end point of the link operation. Targets are assumed to be ready.
	 * @param {Function} callback The callback to call when operation has ended. 
	 * @param {Boolean} callback.success True if the operation was successful
	 * @protected
	 */
	_link: function(sourceTargets, target, callback)
	{
		if (sourceTargets.length == 0)
		{
			return;
		}
		var sourceTarget = sourceTargets[0];

		Ametys.plugins.skineditor.resources.SkinResourcesActions.move (sourceTarget.getParameters().skinName, sourceTarget.getParameters().path, target.getParameters().path);
	},
	
	link: function(source, target, callback, relationType)
	{
		this._link(source.getTargets(), target.getTarget(), callback);
	},
	
});
