/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.skineditor.skin;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.core.right.RightManager;
import org.ametys.core.right.RightManager.RightResult;
import org.ametys.core.ui.Callable;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.util.path.PathUtils;
import org.ametys.plugins.skincommons.AbstractCommonSkinDAO;
import org.ametys.runtime.authentication.AccessDeniedException;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.web.skin.Skin;

/**
 * DAO for files and folders inside a skin directory
 */
public class SkinDAO extends AbstractCommonSkinDAO
{
    /** The Avalon role */
    public static final String ROLE = SkinDAO.class.getName();
    
    /** Constant for the id of right to edit all skins */
    public static final String EDIT_SKINS_RIGHT_ID = "Plugins_SkinEditor_EditAllSkin";
    /** Constant for the id of right to edit teh current skin */
    public static final String EDIT_CURRENT_SKIN_RIGHT_ID = "Plugins_SkinEditor_EditCurrentSkin";
    
    /** Constant for skin editor tool id */
    public static final String SKIN_EDITOR_TOOL_ID = "uitool-skineditor";
    
    private static final String __WORK_MODE = "work";
    private static final String __PROD_MODE = "prod";

    private RightManager _rightManager;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _rightManager = (RightManager) manager.lookup(RightManager.ROLE);
    }
    
    @Override
    protected void checkUserRight(String skinName)
    {
        UserIdentity user = _userProvider.getUser();
        if (!(_rightManager.hasRight(user, EDIT_SKINS_RIGHT_ID, "/${WorkspaceName}") == RightResult.RIGHT_ALLOW
            || _skinsManager.getSkinNameFromRequest().equals(skinName) && _rightManager.hasRight(user, EDIT_CURRENT_SKIN_RIGHT_ID, "/${WorkspaceName}") == RightResult.RIGHT_ALLOW))
        {
            
            throw new AccessDeniedException("User '" + user + "' tried perform operation on skin '" + skinName + "' without sufficient right");
        }
    }
    
    /**
     * Get the list of available and modifiable skins for a site
     * @return the list of skins
     */
    @Callable(rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public List<Object> getSkinsList()
    {
        Set<String> skins = new HashSet<>();
        // Do not trust the client to determine the current site.
        String currentSkinId = _skinsManager.getSkinNameFromRequest();
        
        UserIdentity user = _userProvider.getUser();
        if (_rightManager.hasRight(user, EDIT_SKINS_RIGHT_ID, "/${WorkspaceName}") == RightResult.RIGHT_ALLOW)
        {
            skins.addAll(_skinsManager.getSkins());
        }
        else if (_rightManager.hasRight(user, EDIT_CURRENT_SKIN_RIGHT_ID, "/${WorkspaceName}") == RightResult.RIGHT_ALLOW)
        {
            skins.add(currentSkinId);
        }
        
        return skins.stream()
            .map(id -> _skinsManager.getSkin(id))
            .filter(Skin::isModifiable)
            .filter(Predicate.not(Skin::isAbstract))
            .map(s -> _skin2JsonObject(s, s.getId().equals(currentSkinId)))
            .collect(Collectors.toList());
    }
    
    private Map<String, Object> _skin2JsonObject (Skin skin, boolean current)
    {
        Map<String, Object> jsonObject = new HashMap<>();
        
        I18nizableText label = skin.getLabel();
        String icon = skin.getLargeImage();
        
        jsonObject.put("id", skin.getId());
        jsonObject.put("current", current);
        jsonObject.put("label", label);
        jsonObject.put("icon", icon);
        
        return jsonObject;
    }
    
    /**
     * Open a skin for editing
     * @param skinId the skin id
     * @param mode the edition mode
     * @param unlinkModel True to remove any existing change
     * @return the skin id
     * @throws Exception if an error occurs during the skin opening process
     */
    @Callable(rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public String openSkin(String skinId, String mode, boolean unlinkModel) throws Exception
    {
        checkUserRight(skinId);
        
        Skin skin = _skinsManager.getSkin(skinId);
        if (!skin.isModifiable())
        {
            throw new IllegalStateException("The skin '" + skinId + "' is not modifiable and thus cannot be opened in skin editor.");
        }
        
        Path tempDir = _skinHelper.getTempDirectory(skinId);
        Path workDir = _skinHelper.getWorkDirectory(skinId);
        Path skinDir = _skinHelper.getSkinDirectory(skinId);
        
        if (unlinkModel)
        {
            unlinkModel (skinDir);
            unlinkModel(workDir);
            unlinkModel(tempDir);
        }
        
        if (__PROD_MODE.equals(mode) || __WORK_MODE.equals(mode))
        {
            // Delete temp directory if exists
            if (Files.exists(tempDir))
            {
                _skinHelper.deleteQuicklyDirectory(tempDir);
            }
            
            if (__PROD_MODE.equals(mode))
            {
                // Delete work directory if exists
                if (Files.exists(workDir))
                {
                    _skinHelper.deleteQuicklyDirectory(workDir);
                }
                
                // Copy from skin
                PathUtils.copyDirectory(skinDir, workDir);
            }
                    
            // Copy work in temp
            PathUtils.copyDirectory(workDir, tempDir);
            
            // Create .lock file
            _lockManager.updateLockFile(tempDir, SKIN_EDITOR_TOOL_ID);
        }
        else
        {
            // Update .lock file
            _lockManager.updateLockFile(tempDir, SKIN_EDITOR_TOOL_ID);
        }
        
        return skinId;
    }
    
    /**
     * Unlink model
     * @param skinDir The skin directory
     * @throws IOException If an error occurred
     */
    protected void unlinkModel (Path skinDir) throws IOException
    {
        Path modelFile = skinDir.resolve("model.xml");
        Path bakFile = skinDir.resolve("model.xml.bak");
        
        if (Files.exists(bakFile))
        {
            // Delete old bak file if exists
            PathUtils.deleteQuietly(bakFile);
        }
        
        if (Files.exists(modelFile))
        {
            Files.move(modelFile, bakFile);
        }
    }
}
