/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * DAO for manipulating a skin model
 */
Ext.define("Ametys.plugins.skinfactory.model.SkinModelDAO",
{
    singleton: true,

    constructor: function ()
    {
        /**
         * @callable
         * @member Ametys.plugins.skinfactory.model.SkinModelDAO
         * @method getModel 
         * This calls the method 'getModel' of the server DAO 'org.ametys.skinfactory.model.SkinModelDAO'.
         * Retrieve the informations on a model
         * @param {Object[]} parameters The parameters to transmit to the server method.
         * @param {String} parameters.modelId The id of the model
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.returnedValue The model information (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
         * @param {Object} callback.returnedValue.name The name of the model
         * @param {Object} callback.returnedValue.title The title of the model
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
        {
            role: "org.ametys.skinfactory.model.SkinModelDAO",
            methodName: "getModel",
            waitMessage: true,
            errorMessage: {
                msg: "{{i18n PLUGINS_SKINFACTORY_ADMINISTRATOR_SKINS_MODELS_LIST_ERROR}}",
                category: Ext.getClassName(this)
            }
        });

        /**
         * @callable
         * @member org.ametys.skinfactory.model.SkinModelDAO
         * @method getSkinsAndModels
         * This calls the method 'getSkinsAndModels' of the server DAO 'org.ametys.skinfactory.model.SkinModelDAO'.
         * Retrieve the list of skins and models from the server.
         * @param {Object[]} parameters The parameters to transmit to the server method. Should be an empty array because this method requires
         * @param {Boolean} parameters.includeAbstract True to also includes abstract skins
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope.
         * @param {Object} callback.returnedValue The value return from the server. Null on error (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
         * @param {Object[]} callback.returnedValue.models The list of models availables
         * @param {Object[]} callback.returnedValue.skins The list of skins availables
         * @param {Object} callback.arguments Other arguments specified in option.arguments
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
        {
            role: "org.ametys.skinfactory.model.SkinModelDAO",
            methodName: "getSkinsAndModels",
            waitMessage: true,
            errorMessage: {
                msg: "{{i18n PLUGINS_SKINFACTORY_ADMINISTRATOR_SKINS_MODELS_LIST_ERROR}}",
                category: Ext.getClassName(this)
            }
        });
        
        /**
         * @callable
         * @member Ametys.plugins.skinfactory.model.SkinModelDAO
         * @method importModel 
         * This calls the method 'importModel' of the server DAO 'org.ametys.skinfactory.model.SkinModelDAO'.
         * Import a model from a zip file
         * @param {Object[]} parameters The parameters to transmit to the server method.
         * @param {String} parameters.modelName The model
         * @param {String} parameters.tmpDirPath The temporary directory of the upload
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.returnedValue The name of the new model (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
        {
            role: "org.ametys.skinfactory.model.SkinModelDAO",
            methodName: "importModel",
            callback: {
                handler: this._importModelCb
            },
            waitMessage: true,
            errorMessage: {
                msg: "{{i18n PLUGINS_SKINFACTORY_SKIN_MODEL_IMPORT_ERROR}}",
                category: Ext.getClassName(this)
            }
        });

        /**
         * @callable
         * @member Ametys.plugins.skinfactory.model.SkinModelDAO
         * @method generateSkin 
         * This calls the method 'generateSkin' of the server DAO 'org.ametys.skinfactory.model.SkinModelDAO'.
         * Generate a new skin from a model
         * @param {Object[]} parameters The parameters to transmit to the server method.
         * @param {String} parameters.skinId The new skin id
         * @param {String} parameters.modelId The id of the model used to generate the skin
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.returnedValue An error message, or null on success (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
        {
            role: "org.ametys.skinfactory.model.SkinModelDAO",
            methodName: "generateSkin",
            callback: {
                handler: this._generateSkinCb
            },
            waitMessage: true,
            errorMessage: {
                msg: "{{i18n PLUGINS_SKINFACTORY_SKIN_MODEL_GENERATE_SKIN_ERROR}}",
                category: Ext.getClassName(this)
            }
        });

        /**
         * @callable
         * @member Ametys.plugins.skinfactory.model.SkinModelDAO
         * @method applyModelToAll 
         * This calls the method 'applyModelToAll' of the server DAO 'org.ametys.skinfactory.model.SkinModelDAO'.
         * Reapply the specified model to all the skins associated with it.
         * @param {Object[]} parameters The parameters to transmit to the server method.
         * @param {String} parameters.modelId The id of the model to reapply
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {String} callback.returnedValue The list of modified skins. (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
        {
            role: "org.ametys.skinfactory.model.SkinModelDAO",
            methodName: "applyModelToAll",
            callback: {
                handler: this._applyModelToAllCb
            },
            waitMessage: true,
            errorMessage: {
                msg: "{{i18n PLUGINS_SKINFACTORY_SKIN_MODEL_APPLY_ALL_ERROR}}",
                category: Ext.getClassName(this)
            }
        });

        /**
         * @callable
         * @member Ametys.plugins.skinfactory.model.SkinModelDAO
         * @method applyModel 
         * This calls the method 'applyModel' of the server DAO 'org.ametys.skinfactory.model.SkinModelDAO'.
         * Apply a model to the skin.
         * @param {Object[]} parameters The parameters to transmit to the server method.
         * @param {String} parameters.skinId The id of the skin
         * @param {String} parameters.modelId The id of the model to apply
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
        {
            role: "org.ametys.skinfactory.model.SkinModelDAO",
            methodName: "applyModel",
            callback: {
                handler: this._applyModelCb
            },
            waitMessage: true,
            errorMessage: {
                msg: "{{i18n PLUGINS_SKINFACTORY_SKIN_MODEL_APPLY_ERROR}}",
                category: Ext.getClassName(this)
            }
        });

        /**
         * @callable
         * @member Ametys.plugins.skinfactory.model.SkinModelDAO
         * @method delete 
         * This calls the method 'delete' of the server DAO 'org.ametys.skinfactory.model.SkinModelDAO'.
         * Delete an existing model
         * @param {Object[]} parameters The parameters to transmit to the server method.
         * @param {String} parameters.modelId The id of the model to delete
         * @param {Ametys.message.MessageTarget} parameters.target The model target
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
        {
            role: "org.ametys.skinfactory.model.SkinModelDAO",
            methodName: "delete",
            localParamsIndex: 1,
            callback: {
                handler: this._deleteCb
            },
            waitMessage: true,
            errorMessage: {
                msg: "{{i18n PLUGINS_SKINFACTORY_SKIN_MODEL_DELETE_ERROR}}",
                category: Ext.getClassName(this)
            }
        });

        /**
         * @callable
         * @member Ametys.plugins.skinfactory.model.SkinModelDAO
         * @method unlinkModel 
         * This calls the method 'unlinkModel' of the server DAO 'org.ametys.skinfactory.model.SkinModelDAO'.
         * Delete a skin
         * @param {Object[]} parameters The parameters to transmit to the server method.
         * @param {String} parameters.id The id of the skin
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {String} callback.returnedValue The name of the skin. (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
        {
            role: "org.ametys.skinfactory.model.SkinModelDAO",
            methodName: "unlinkModel",
            callback: {
                handler: this._unlinkModelCb
            },
            waitMessage: true,
            errorMessage: {
                msg: "{{i18n PLUGINS_SKINFACTORY_SKIN_MODEL_UNLINK_ERROR}}",
                category: Ext.getClassName(this)
            }
        });
        
        /**
         * @callable
         * @member Ametys.plugins.skinfactory.model.SkinModelDAO
         * @method modelExists 
         * This calls the method 'modelExists' of the server DAO 'org.ametys.skinfactory.model.SkinModelDAO'.
         * Determines if a model exists.
         * @param {Object[]} parameters The parameters to transmit to the server method.
         * @param {String} parameters.id The id of the model
         * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Boolean} callback.exists True if the model exists, false otherwise.
         * @param {Object} callback.arguments Other arguments specified in option.arguments                 
         * @param {Object} [options] Advanced options for the call.
         * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
         * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
         * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
         * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
         * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
         * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
         * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
         */
        this.addCallables(
        {
            role: "org.ametys.skinfactory.model.SkinModelDAO",
            methodName: "modelExists",
            errorMessage: "{{i18n PLUGINS_SKINFACTORY_SKIN_MODEL_IMPORT_TEST_ERROR}}"
        });
    },

    /**
     * Callback after importing a skin. Send a CREATED Message
     * @param {String} modelId The id of the model
     * @private
     */
    _importModelCb: function(modelId)
    {
        Ext.create("Ametys.message.Message",
        {
            type: Ametys.message.Message.CREATED,
            targets:
            {
                id: Ametys.message.MessageTarget.SKINMODEL,
                parameters:
                {
                    id: modelId
                }
            }
        });
    },

    /**
     * Callback after importing a skin. Send a CREATED Message
     * @param {String} error The error code, or null if successful
     * @param {Object} args The arguments
     * @param {Object} params The initial parameters
     * @private
     */
    _generateSkinCb: function(error, args, params)
    {
    	if (error && error == "already-exists")
        {
            Ametys.Msg.show({
                title: "{{i18n plugin.web:PLUGINS_WEB_ADMINISTRATOR_SKINS_ALREADY_EXIST_TITLE}}",
                msg: "{{i18n plugin.web:PLUGINS_WEB_ADMINISTRATOR_SKINS_ALREADY_EXIST}}",
                buttons: Ext.Msg.OK,
                icon: Ext.MessageBox.ERROR
            });
        }
    	else if (!error)
        {
            Ext.create("Ametys.message.Message",
            {
                type: Ametys.message.Message.CREATED,
                targets:
                {
                    id: Ametys.message.MessageTarget.SKIN,
                    parameters:
                    {
                        id: params[0]
                    }
                }
            });
        }
    },

    /**
     * Callback after applying model. Send a MODIFIED Message
     * @param {Object} result The result
     * @param {String[]} result.modifiedSkins The list of modified skins' ids
     * @param {String[]} result.unmodifiedSkins The list of unmodified skins' ids
     * @private
     */
    _applyModelToAllCb: function(result)
    {
        var modifiedSkins = result.modifiedSkins || [];
        var unmodifiedSkins = result.unmodifiedSkins || [];
        var unmodifiableSkins = result.unmodifiableSkins || [];
        
        Ext.Array.forEach(modifiedSkins, function(skin) {
            Ext.create("Ametys.message.Message",
            {
                type: Ametys.message.Message.MODIFIED,
                targets:
                {
                    id: Ametys.message.MessageTarget.SKIN,
                    parameters:
                    {
                        id: skin.name
                    }
                }
            });
        });
        
        if (unmodifiedSkins.length > 0)
        {
            var skinLabels = [];
            Ext.Array.each(unmodifiedSkins, function (skin) {
                skinLabels.push(skin.label + ' (' + skin.name + ')');
            });
            
            Ametys.Msg.show({
                title: "{{i18n PLUGINS_SKINFACTORY_SKIN_MODEL_APPLY_ALL_LABEL}}",
                msg: Ext.String.format("{{i18n PLUGINS_SKINFACTORY_SKIN_MODEL_APPLY_SKINS_IN_USE_ERROR}}", skinLabels.join(", ")),
                buttons: Ext.Msg.OK,
                icon: Ext.MessageBox.ERROR
            });
        }
        
        if (unmodifiableSkins.length > 0)
        {
            var skinLabels = [];
            Ext.Array.each(unmodifiableSkins, function (skin) {
                skinLabels.push(skin.label + ' (' + skin.name + ')');
            });
            
            Ametys.Msg.show({
                title: "{{i18n PLUGINS_SKINFACTORY_SKIN_MODEL_APPLY_ALL_LABEL}}",
                msg: Ext.String.format("{{i18n PLUGINS_SKINFACTORY_SKIN_MODEL_APPLY_SKINS_UNMODIFIABLE_ERROR}}", skinLabels.join(", ")),
                buttons: Ext.Msg.OK,
                icon: Ext.MessageBox.WARNING
            });
        }
        
    },

    /**
     * Callback after importing a skin. Send a MODIFIED Message
     * @param {Boolean} success true if the skin was successfully updated
     * @param {Object} args The arguments
     * @param {Object} params The initial parameters
     * @private
     */
    _applyModelCb: function(success, args, params)
    {
        if (success)
        {
            Ext.create("Ametys.message.Message",
	        {
	            type: Ametys.message.Message.MODIFIED,
	            targets:
	            {
	                id: Ametys.message.MessageTarget.SKIN,
	                parameters:
	                {
	                    id: params[0]
	                }
	            }
	        });
        }
        else
        {
            Ametys.Msg.show({
                title: "{{i18n PLUGINS_SKINFACTORY_SKIN_MODEL_APPLY_LABEL}}",
                msg: "{{i18n PLUGINS_SKINFACTORY_SKIN_MODEL_APPLY_SKIN_IN_USE_ERROR}}",
                buttons: Ext.Msg.OK,
                icon: Ext.MessageBox.ERROR
            });
        }
        
    },

    /**
     * Callback after deleting a model. Send a DELETE Message
     * @param {String} response The response from the server
     * @param {Object} args The arguments
     * @param {Object} params The initial parameters
     * @private
     */
    _deleteCb: function(response, args, params)
    {
    	Ext.create("Ametys.message.Message",
        {
            type: Ametys.message.Message.DELETED,
            targets: params[1]
        });
    },

    /**
     * Callback after unlinking skin from its model. Send a MODIFIED Message on skin
     *	@param {String} response The response from the server
     * @param {Object} args The arguments
     * @param {Object} params The initial parameters
     * @private
     */
    _unlinkModelCb: function(response, args, params)
    {
        Ext.create("Ametys.message.Message",
        {
            type: Ametys.message.Message.MODIFIED,
            targets:
            {
                id: Ametys.message.MessageTarget.SKIN,
                parameters:
                {
                	id: params[0]
                }
            }
        });
    }
    
    
});
