/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.skinfactory.clientsidelement;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.configuration.DefaultConfiguration;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.cms.languages.Language;
import org.ametys.cms.languages.LanguagesManager;
import org.ametys.core.ui.ClientSideElement;
import org.ametys.core.ui.SimpleMenu;
import org.ametys.core.ui.StaticClientSideElement;
import org.ametys.core.util.I18nUtils;
import org.ametys.runtime.i18n.I18nizableText;

/**
 * {@link StaticClientSideElement} for selection of language for a skin
 */
public class SkinLanguageMenu extends SimpleMenu
{
    private LanguagesManager _languageManager;
    private boolean _languagesInitialized;
    private I18nUtils _i18nUtils;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _languageManager = (LanguagesManager) smanager.lookup(LanguagesManager.ROLE);
        _i18nUtils = (I18nUtils) smanager.lookup(I18nUtils.ROLE);
    }
    
    @Override
    protected void _getGalleryItems(Map<String, Object> parameters, Map<String, Object> contextualParameters)
    {
        try
        {
            _lazyInitializeLanguageGallery();
        }
        catch (Exception e)
        {
            throw new IllegalStateException("Unable to lookup client side element local components", e);
        }
        
        if (_galleryItems.size() > 0)
        {
            parameters.put("gallery-item", new LinkedHashMap<>());
            
            for (GalleryItem galleryItem : _galleryItems)
            {
                @SuppressWarnings("unchecked")
                Map<String, Object> galleryItems = (Map<String, Object>) parameters.get("gallery-item");
                galleryItems.put("gallery-groups", new ArrayList<>());
                
                for (GalleryGroup galleryGp : galleryItem.getGroups())
                {
                    @SuppressWarnings("unchecked")
                    List<Object> galleryGroups = (List<Object>) galleryItems.get("gallery-groups");
                    
                    Map<String, Object> groupParams = new LinkedHashMap<>();
                    
                    I18nizableText label = galleryGp.getLabel();
                    groupParams.put("label", label);
                    
                    // Group items
                    List<String> gpItems = new ArrayList<>();
                    for (ClientSideElement element : galleryGp.getItems())
                    {
                        gpItems.add(element.getId());
                    }
                    
                    if (gpItems.size() > 0)
                    {
                        groupParams.put("items", gpItems);
                        galleryGroups.add(groupParams);
                    }
                }
            }
        }
    }
    
    /**
     * Lazy initialization of the language gallery
     * @throws ConfigurationException if the language configuration is wrong
     */
    private synchronized void _lazyInitializeLanguageGallery() throws ConfigurationException
    {
        if (!_languagesInitialized)
        {
            Map<String, Language> availableLanguages = _languageManager.getAvailableLanguages();
            
            if (availableLanguages.size() > 0)
            {
                GalleryItem galleryItem = new GalleryItem();
                
                GalleryGroup galleryGroup = new GalleryGroup(new I18nizableText("plugin." + _pluginName, "RIBBON_TABS_TAB_SKINFACTORY_GROUP_LANG_LABEL"));
                galleryItem.addGroup(galleryGroup);
                
                for (String id : availableLanguages.keySet())
                {
                    Language language = availableLanguages.get(id);
                    
                    String itemId = this.getId() + "-" + id;
                    
                    Configuration conf = _getLanguageConfiguration(itemId, language);
                    _getGalleryItemManager().addComponent(_pluginName, null, itemId, StaticClientSideElement.class, conf);
                    galleryGroup.addItem(new UnresolvedItem(itemId, true));
                }
                
                _galleryItems.add(galleryItem);
            }
            
            if (_galleryItems.size() > 0)
            {
                try
                {
                    _getGalleryItemManager().initialize();
                }
                catch (Exception e)
                {
                    throw new ConfigurationException("Unable to lookup parameter local components", e);
                }
            }
        }
        
        _languagesInitialized = true;
    }
    
    /**
     * Get the configuration of the language item
     * @param id The id of item
     * @param language The language
     * @return The configuration
     */
    protected Configuration _getLanguageConfiguration (String id, Language language)
    {
        DefaultConfiguration conf = new DefaultConfiguration("extension");
        conf.setAttribute("id", id);
        
        DefaultConfiguration classConf = new DefaultConfiguration("class");
        classConf.setAttribute("name", "Ametys.ribbon.element.ui.ButtonController");
        
        // Label
        DefaultConfiguration labelConf = new DefaultConfiguration("label");
        labelConf.setValue(_i18nUtils.translate(language.getLabel()));
        classConf.addChild(labelConf);
        
        // Icons
        DefaultConfiguration iconSmallConf = new DefaultConfiguration("icon-small");
        iconSmallConf.setValue(language.getSmallIcon());
        classConf.addChild(iconSmallConf);
        DefaultConfiguration iconMediumConf = new DefaultConfiguration("icon-medium");
        iconMediumConf.setValue(language.getMediumIcon());
        classConf.addChild(iconMediumConf);
        DefaultConfiguration iconLargeConf = new DefaultConfiguration("icon-large");
        iconLargeConf.setValue(language.getLargeIcon());
        classConf.addChild(iconLargeConf);

        // Language
        DefaultConfiguration langConf = new DefaultConfiguration("lang");
        langConf.setValue(language.getCode());
        classConf.addChild(langConf);

        // Common configuration
        @SuppressWarnings("unchecked")
        Map<String, Object> commonConfig = (Map<String, Object>) this._script.getParameters().get("items-config");
        for (String tagName : commonConfig.keySet())
        {
            DefaultConfiguration c = new DefaultConfiguration(tagName);
            c.setValue(String.valueOf(commonConfig.get(tagName)));
            classConf.addChild(c);
        }
        
        conf.addChild(classConf);
        return conf;
    }
    
}
