/*
 *  Copyright 2011 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.skinfactory.clientsidelement;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.configuration.DefaultConfiguration;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;

import org.ametys.core.ui.SimpleMenu;
import org.ametys.core.ui.StaticClientSideElement;
import org.ametys.core.util.I18nUtils;
import org.ametys.web.skin.SkinModel;
import org.ametys.web.skin.SkinModel.Theme;
import org.ametys.web.skin.SkinModelsManager;

/**
 * {@link StaticClientSideElement} for thematic of colors
 *
 */
public class ThemeColorsMenu extends SimpleMenu
{
    private SkinModelsManager _modelsManager;
    private I18nUtils _i18nUtils;
    private boolean _themeColorsInitialized;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _modelsManager = (SkinModelsManager) smanager.lookup(SkinModelsManager.ROLE);
        _i18nUtils = (I18nUtils) smanager.lookup(I18nUtils.ROLE);
    }
    
    @Override
    public List<Script> getScripts(boolean ignoreRights, Map<String, Object> contextParameters)
    {
        Map<String, String> rights = getRights(contextParameters);
        
        if (!ignoreRights && !hasRight(rights))
        {
            return Collections.EMPTY_LIST;
        }
        
        try
        {
            Set<String> models = _modelsManager.getModels();
            if (models.size() > 0)
            {
                _lazyInitializeThemeColorsMenu(contextParameters);
                return super.getScripts(ignoreRights, contextParameters);
            }
            else
            {
                return Collections.EMPTY_LIST;
            }
        }
        catch (Exception e)
        {
            throw new IllegalStateException("Unable to lookup client side element local components", e);
        }
    }

    /**
     * Lazy initialization of the theme colors menu
     * @param contextualParameters The contextual parameters
     * @throws ConfigurationException if the theme colors' configuration is wrong
     * @throws ProcessingException if an error occurs when retrieving the models 
     */
    private synchronized void _lazyInitializeThemeColorsMenu(Map<String, Object> contextualParameters) throws ConfigurationException, ProcessingException
    {
        if (!_themeColorsInitialized)
        {
            Set<String> models = _modelsManager.getModels();
            
            if (models.size() > 0)
            {
                for (String modelName : models)
                {
                    SkinModel model = _modelsManager.getModel(modelName);
                    Map<String, Theme> themes = model.getThemes();
                
                    if (themes.size() > 0)
                    {
                        for (Theme theme : themes.values())
                        {
                            String itemId = this.getId() + "-" + modelName + "-" + theme.getId();
                            
                            Configuration conf = _getThemeColorsConfiguration(itemId, theme, modelName, contextualParameters);
                            _menuItemManager.addComponent(_pluginName, null, itemId, StaticClientSideElement.class, conf);
                            
                            _unresolvedMenuItems.add(new UnresolvedItem(itemId, true));
                        }
                    }
                }
            }
            
            _themeColorsInitialized = true;
        }
    }

    /**
     * Get the configuration of the model item
     * @param id The id of item
     * @param theme The design
     * @param modelName The model name
     * @param contextualParameters The contextual parameters
     * @return The configuration
     */
    protected Configuration _getThemeColorsConfiguration(String id, Theme theme, String modelName, Map<String, Object> contextualParameters)
    {
        DefaultConfiguration conf = new DefaultConfiguration("extension");
        conf.setAttribute("id", id);
        
        DefaultConfiguration classConf = new DefaultConfiguration("class");
        classConf.setAttribute("name", "Ametys.plugins.skinfactory.ribbon.menu.ThemeColorsMenuItem");
        
        // Label
        DefaultConfiguration labelConf = new DefaultConfiguration("label");
        labelConf.setValue(_i18nUtils.translate(theme.getLabel()));
        classConf.addChild(labelConf);
        
        // Colors
        List<String> colors = theme.getColors();
        DefaultConfiguration colorSizeConf = new DefaultConfiguration("color-size");
        colorSizeConf.setValue(Integer.toString(colors.size()));
        classConf.addChild(colorSizeConf);
        
        int j = 0;
        for (String color : colors)
        {
            DefaultConfiguration colorConf = new DefaultConfiguration("color-" + j);
            colorConf.setValue(color);
            classConf.addChild(colorConf);
            j++;
        }
        
        // Model name
        DefaultConfiguration modelNameConf = new DefaultConfiguration("modelName");
        modelNameConf.setValue(modelName);
        classConf.addChild(modelNameConf);
        
        // Theme
        DefaultConfiguration themeConf = new DefaultConfiguration("theme");
        themeConf.setValue(theme.getId());
        classConf.addChild(themeConf);
        
        // Common configuration
        @SuppressWarnings("unchecked")
        Map<String, Object> commonConfig = (Map<String, Object>) this._script.getParameters().get("items-config");
        for (String tagName : commonConfig.keySet())
        {
            DefaultConfiguration c = new DefaultConfiguration(tagName);
            c.setValue(String.valueOf(commonConfig.get(tagName)));
            classConf.addChild(c);
        }
        
        conf.addChild(classConf);
        
        Map<String, String> rights = getRights(contextualParameters);
        List<DefaultConfiguration> rightsList = new ArrayList<> ();
        for (Map.Entry<String, String> rightAndPrefix : rights.entrySet())
        {
            DefaultConfiguration rightConf = new DefaultConfiguration("right");
            rightConf.setValue(rightAndPrefix.getKey());
            if (rightAndPrefix.getValue() != null)
            {
                rightConf.setAttribute("context-prefix", rightAndPrefix.getValue());
            }
            rightsList.add(rightConf);
        }
        
        if (rightsList.size()  == 1)
        {
            // right
            conf.addChild(rightsList.get(0));
        }
        else if (rightsList.size() > 1)
        {
            // rights
            DefaultConfiguration rightsConf = new DefaultConfiguration("rights");
            for (DefaultConfiguration rightConf : rightsList)
            {
                rightsConf.addChild(rightConf);
            }
            conf.addChild(rightsConf);
        }
        
        return conf;
    }
    

}
