/*
 *  Copyright 2011 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.skinfactory.parameters;

import java.nio.file.Path;
import java.util.Map;

import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.web.skin.SkinModel;

/**
 * Class representing a parameter of a skin.
 */
public abstract class AbstractSkinParameter
{
    String _id;
    I18nizableText _label;
    I18nizableText _description;
    
    /** Enumeration of supported types */
    public static enum SkinParameterType
    {
        /** Image */
        IMAGE,
        /** CSS property */
        CSS,
        /** Variant */
        VARIANT,
        /** Text */
        TEXT,
        /** I18n text */
        I18N
    }
    
    /**
     * Constructor
     * @param id the unique id
     * @param label the label
     * @param description the description
     */
    public AbstractSkinParameter(String id, I18nizableText label, I18nizableText description)
    {
        this._id = id;
        this._label = label;
        this._description = description;
    }
    
    /**
     * Get the unique id
     * @return the unique id
     */
    public String getId()
    {
        return _id;
    }
    
    /**
     * Get the label
     * @return the label
     */
    public I18nizableText getLabel()
    {
        return _label != null ? _label : new I18nizableText(_id);
    }

    /**
     * Set the label
     * @param label the label to set
     */
    public void setLabel (I18nizableText label)
    {
        _label = label;
    }
    
    /**
     * Get the description
     * @return the description
     */
    public I18nizableText getDescription()
    {
        return _description != null ? _description : new I18nizableText("");
    }
    
    /**
     * Set the description
     * @param description the description to set
     */
    public void setDescription (I18nizableText description)
    {
        _description = description;
    }
    
    /**
     * Get type
     * @return The type
     */
    public abstract SkinParameterType getType();
    
    /**
     * Apply parameter to the skin
     * @param tempDir the work directory
     * @param modelDir the model directory
     * @param value the value
     * @param lang The language. Can be null.
     */
    public abstract void apply(Path tempDir, Path modelDir, Object value, String lang);
    
    /**
     * SAX the parameter
     * @param contentHandler The content handler to SAX into
     * @param modelName The model name
     * @throws SAXException if an error occurred while SAXing 
     */
    public abstract void toSAX (ContentHandler contentHandler, String modelName) throws SAXException;

    /**
     * Transform the parameter into a JSON object
     * @param modelName The model name
     * @return The JSON representation of the parameter
     */
    public abstract Map<String, Object> toJson (String modelName);
    
    /**
     * Get the default value
     * @param model The model
     * @return The default value
     */
    public abstract Object getDefaultValue (SkinModel model);
    
    /**
     * Get the default value
     * @param model The model
     * @param lang The language
     * @return The default value
     */
    public abstract Object getDefaultValue (SkinModel model, String lang);
    
    /**
     * Widget class
     *
     */
    public class Widget 
    {
        String _group;
        String[] _params;
        String _name;
        
        /**
         * Constructor
         * @param name the name of the widget
         * @param group the group of the widget
         * @param params the widget parameters
         */
        public Widget(String name, String group, String[] params)
        {
            _name = name;
            _group = group;
            _params = params;
        }
    }
}
