/*
 *  Copyright 2011 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.skinfactory.parameters;

import java.io.IOException;
import java.io.OutputStream;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.HashMap;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.io.IOUtils;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.web.skin.SkinModel;

/**
 * Implementation of {@link AbstractSkinParameter} for a i18niable text
 */
public class I18nizableTextParameter extends AbstractSkinParameter
{
    private String _i18nKey;
    private Map<String, String> _defaultValues;
    
    /**
     * Constructor
     * @param id the unique id
     * @param label the label
     * @param description the description
     * @param i18nKey the i18n key
     * @param defaultValues the default values
     */
    public I18nizableTextParameter(String id, I18nizableText label, I18nizableText description, String i18nKey, Map<String, String> defaultValues)
    {
        super(id, label, description);
        _i18nKey = i18nKey;
        _defaultValues = defaultValues;
    }
    
    @Override
    public SkinParameterType getType()
    {
        return SkinParameterType.I18N;
    }
    
    /**
     * Get the i18n key
     * @return the i18n key
     */
    public String geti18nKey()
    {
        return _i18nKey;
    }

    /**
     * Set the i18n key
     * @param i18nKey the i18n key to set
     */
    public void setKey (String i18nKey)
    {
        _i18nKey = i18nKey;
    }
    
    @Override
    public void apply(Path tempDir, Path modelDir, Object value, String lang)
    {
        Path i18nFile = tempDir.resolve("i18n/messages.xml");
        if (lang != null)
        {
            Path f = tempDir.resolve("i18n/messages_" + lang + ".xml");
            if (Files.exists(f))
            {
                i18nFile = f;
            }
        }
        
        String string = _filetoString (i18nFile);
        
        Pattern pattern = null;
        if (i18nFile.getFileName().toString().equals("messages.xml"))
        {
            pattern = Pattern.compile("^\\s*<message key=\"" + geti18nKey() + "\">([^<]*)<!--\\s*AMETYS\\s*\\(\\s*\"([^\"]+)\"\\s*(?:,\\s*([^,\"\\s]+|\"[^\"]*\")\\s*)?(?:,\\s*([^,\"\\s]+|\"[^\"]*\")\\s*)?\\)\\s*--></message>\\s*$", Pattern.MULTILINE); 
            
        }
        else
        {
            pattern = Pattern.compile("^\\s*<message key=\"" + geti18nKey() + "\">([^<]*)</message>\\s*$", Pattern.MULTILINE); 
        }
        
        Matcher matcher = pattern.matcher(string);
        
        try (OutputStream os = Files.newOutputStream(i18nFile))
        {
            int startIndex = 0;
            
            while (matcher.find())
            {
                int propertyStart = matcher.start(1);
                
                String beforeValue = string.substring(startIndex, propertyStart);
                IOUtils.write(beforeValue, os, "UTF-8");
                IOUtils.write((String) value, os, "UTF-8");
                
                startIndex = matcher.end(1);
            }
            
            IOUtils.write(string.substring(startIndex), os, "UTF-8");
        }
        catch (IOException e)
        {
            throw new SkinParameterException ("Unable to apply css parameter '" + getId() + "'", e);
        }
    }
    
    private String _filetoString(Path file)
    {
        try
        {
            return Files.readString(file, StandardCharsets.UTF_8);
        }
        catch (IOException e)
        {
            throw new SkinParameterException ("Unable to parse file '" + file.getFileName().toString() + "'", e);
        }
    }
    
    @Override
    public void toSAX(ContentHandler contentHandler, String modelName) throws SAXException
    {
        AttributesImpl attrs = new AttributesImpl();
        attrs.addCDATAAttribute("id", getId());
        attrs.addCDATAAttribute("type", SkinParameterType.I18N.name().toLowerCase());
        
        XMLUtils.startElement(contentHandler, "parameter", attrs);
        
        getLabel().toSAX(contentHandler, "label");
        getDescription().toSAX(contentHandler, "description");
        XMLUtils.createElement(contentHandler, "i18nkey", geti18nKey());
        
        XMLUtils.endElement(contentHandler, "parameter");
    }
    
    @Override
    public Map<String, Object> toJson(String modelName)
    {
        Map<String, Object> jsonObject = new HashMap<>();
            
        jsonObject.put("id", getId());
        jsonObject.put("type", SkinParameterType.I18N.name().toLowerCase());
        jsonObject.put("label", getLabel());
        jsonObject.put("description", getDescription());
        jsonObject.put("i18nkey", geti18nKey());
        
        return jsonObject;
    }
    
    /**
     * Get default values
     * @return The default values
     */
    public Map<String, String> getDefaultValues ()
    {
        return _defaultValues;
    }
    
    @Override
    public String getDefaultValue(SkinModel model)
    {
        return getDefaultValue (model, null);
    }
    
    @Override
    public String getDefaultValue (SkinModel model, String lang)
    {
        if (lang != null && _defaultValues.containsKey(lang))
        {
            return _defaultValues.get(lang);
        }
        
        return _defaultValues.entrySet().iterator().next().getValue();
    }
}
