/*
 *  Copyright 2011 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.skinfactory.parameters;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Stream;

import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.core.util.path.PathUtils;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.skinfactory.filefilter.FileFilter;
import org.ametys.web.skin.SkinModel;

/**
 * Implementation of {@link AbstractSkinParameter} for a variant
 */
public class VariantParameter extends AbstractSkinParameter
{
    private String _iconGlyph;
    private String _iconSmall;
    private String _iconLarge;
    private List<Variant> _variants;
    
    
    /**
     * Constructor
     * @param id the unique id
     * @param label the label
     * @param description the description
     * @param variants the list of values for this variant
     */
    public VariantParameter(String id, I18nizableText label, I18nizableText description, List<Variant> variants)
    {
        super(id, label, description);
        _variants = variants;
    }
    
    /**
     * Constructor
     * @param id the unique id
     * @param label the label
     * @param description the description
     * @param iconGlyph The CSS classe for icon, to use instead of small and large icon
     * @param iconSmall The small icon
     * @param iconLarge The large icon
     * @param variants the list of values for this variant
     */
    public VariantParameter(String id, I18nizableText label, I18nizableText description, String iconGlyph, String iconSmall, String iconLarge, List<Variant> variants)
    {
        super(id, label, description);
        _iconGlyph = iconGlyph;
        _iconSmall = iconSmall;
        _iconLarge = iconLarge;
        _variants = variants;
    }
    
    @Override
    public SkinParameterType getType()
    {
        return SkinParameterType.VARIANT;
    }
    /**
     * Get the variants
     * @return The variants
     */
    public List<Variant> getVariants()
    {
        return _variants;
    }
    
    /**
     * Set the CSS icon
     * @param iconGlyph the CSS icon
     */
    public void setIconGlyph(String iconGlyph)
    {
        _iconGlyph = iconGlyph;
    }
    
    /**
     * Get the CSS icon
     * @return The CSS icon
     */
    public String getIconGlyph ()
    {
        return _iconGlyph;
    }
    
    /**
     * Set the small icon relative path
     * @param iconSmall the small icon
     */
    public void setIconSmall(String iconSmall)
    {
        _iconSmall = iconSmall;
    }
    
    /**
     * Get the small icon
     * @return The small icon
     */
    public String getIconSmall ()
    {
        return _iconSmall;
    }
    
    /**
     * Set the large icon relative path
     * @param iconLarge the large icon
     */
    public void setIconLarge(String iconLarge)
    {
        _iconLarge = iconLarge;
    }
    
    /**
     * Get the large icon
     * @return The large icon
     */
    public String getIconLarge ()
    {
        return _iconLarge;
    }
    
    @Override
    public void apply(Path tempDir, Path modelDir, Object value, String lang) throws SkinParameterException
    {
        Path variantFile = modelDir.resolve("model/variants/" + this._id);
        Path srcFile = variantFile.resolve((String) value);
        
        try
        {
            // Copy variant files recursively excluding CVS and .svn directories
            PathUtils.copyDirectory(srcFile, tempDir, FileFilter.getModelVariantFilter((String) value, srcFile), false);
        }
        catch (IOException e)
        {
            throw new SkinParameterException ("Unable to apply variant parameter '" + getId() + "'", e);
        }
    }

    @Override
    public void toSAX(ContentHandler contentHandler, String modelName) throws SAXException
    {
        AttributesImpl attrs = new AttributesImpl();
        attrs.addCDATAAttribute("id", getId());
        attrs.addCDATAAttribute("type", SkinParameterType.VARIANT.name().toLowerCase());
        
        XMLUtils.startElement(contentHandler, "parameter", attrs);
        
        getLabel().toSAX(contentHandler, "label");
        getDescription().toSAX(contentHandler, "description");
        
        if (getIconGlyph() != null)
        {
            XMLUtils.createElement(contentHandler, "iconGlyph", getIconGlyph());
        }
        
        if (getIconSmall() != null)
        {
            XMLUtils.createElement(contentHandler, "iconSmall", "/plugins/skinfactory/" + modelName + "/_thumbnail/16/16/model/variants/" + this._id + "/" + getIconSmall());
        }
        else
        {
            XMLUtils.createElement(contentHandler, "iconSmall", "/plugins/skinfactory/resources/img/variant_default_16.png");
        }
        
        if (getIconLarge() != null)
        {
            XMLUtils.createElement(contentHandler, "iconLarge", "/plugins/skinfactory/" + modelName + "/_thumbnail/32/32/model/variants/" + this._id + "/" + getIconLarge());
        }
        else
        {
            XMLUtils.createElement(contentHandler, "iconLarge", "/plugins/skinfactory/resources/img/variant_default_32.png");
        }
        
        XMLUtils.endElement(contentHandler, "parameter");
        
    }

    @Override
    public Map<String, Object> toJson(String modelName)
    {
        Map<String, Object> jsonObject = new HashMap<>();
        
        jsonObject.put("id", getId());
        jsonObject.put("type", SkinParameterType.VARIANT.name().toLowerCase());
        
        jsonObject.put("label", getLabel());
        jsonObject.put("description", getDescription());
        
        if (getIconGlyph() != null)
        {
            jsonObject.put("iconGlyph", getIconGlyph());
        }
        
        if (getIconSmall() != null)
        {
            jsonObject.put("iconSmall", "/plugins/skinfactory/" + modelName + "/_thumbnail/16/16/model/variants/" + this._id + "/" + getIconSmall());
        }
        else
        {
            jsonObject.put("iconSmall", "/plugins/skinfactory/resources/img/variant_default_16.png");
        }
        
        if (getIconLarge() != null)
        {
            jsonObject.put("iconLarge", "/plugins/skinfactory/" + modelName + "/_thumbnail/32/32/model/variants/" + this._id + "/" + getIconLarge());
        }
        else
        {
            jsonObject.put("iconLarge", "/plugins/skinfactory/resources/img/variant_default_32.png");
        }
        
        return jsonObject;
    }
    
    @Override
    public String getDefaultValue(SkinModel model)
    {
        try
        {
            Path variantFile = model.getPath().resolve("model/variants/" + this._id);
            try (Stream<Path> files = Files.list(variantFile))
            {
                return files
                        .filter(FileFilter.getSkinDirectoryFilter())
                        .findFirst()
                        .map(f -> f.getFileName().toString())
                        .orElse(null);
            }
            
        }
        catch (IOException e)
        {
            throw new RuntimeException("Cannot get default value for variant " + this._id + " of model " + model.getId(), e);
        }
    }
    
    @Override
    public String getDefaultValue(SkinModel model, String lang)
    {
        return getDefaultValue(model);
    }
}
