/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.skinfactory.skins;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.stream.Stream;

import javax.xml.transform.TransformerConfigurationException;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathExpressionException;
import javax.xml.xpath.XPathFactory;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

import org.ametys.cms.languages.Language;
import org.ametys.cms.languages.LanguagesManager;
import org.ametys.core.right.RightManager;
import org.ametys.core.right.RightManager.RightResult;
import org.ametys.core.ui.Callable;
import org.ametys.core.upload.Upload;
import org.ametys.core.upload.UploadManager;
import org.ametys.core.user.User;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.user.UserManager;
import org.ametys.core.util.I18nUtils;
import org.ametys.core.util.LambdaUtils;
import org.ametys.core.util.LambdaUtils.LambdaException;
import org.ametys.core.util.path.PathUtils;
import org.ametys.plugins.skincommons.AbstractCommonSkinDAO;
import org.ametys.runtime.authentication.AccessDeniedException;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.skinfactory.SkinFactoryComponent;
import org.ametys.skinfactory.filefilter.FileFilter;
import org.ametys.skinfactory.model.ModelDesignsManager;
import org.ametys.skinfactory.parameters.AbstractSkinParameter;
import org.ametys.skinfactory.parameters.I18nizableTextParameter;
import org.ametys.skinfactory.parameters.ImageParameter;
import org.ametys.skinfactory.parameters.ImageParameter.FileValue;
import org.ametys.skinfactory.parameters.SkinParameterException;
import org.ametys.skinfactory.parameters.Variant;
import org.ametys.skinfactory.parameters.VariantParameter;
import org.ametys.web.repository.site.Site;
import org.ametys.web.skin.Skin;
import org.ametys.web.skin.SkinModel;
import org.ametys.web.skin.SkinModel.CssMenuItem;
import org.ametys.web.skin.SkinModel.CssStyleItem;
import org.ametys.web.skin.SkinModel.Separator;
import org.ametys.web.skin.SkinModel.Theme;
import org.ametys.web.skin.SkinModelsManager;

/**
 * Component to interact with a skin
 */
public class SkinDAO extends AbstractCommonSkinDAO
{
    /** Constant for skin editor tool id */
    public static final String SKIN_FACTORY_TOOL_ID = "uitool-skinfactory";
    
    private static final DateFormat _DATE_FORMAT = new SimpleDateFormat("yyyyMMdd-HHmm");
    
    private static final String __WORK_MODE = "work";
    private static final String __PROD_MODE = "prod";
    
    private I18nUtils _i18nUtils;
    private LanguagesManager _languageManager;
    private ModelDesignsManager _designsManager;
    private SkinFactoryComponent _skinFactoryManager;
    private SkinModelsManager _modelsManager;
    private UploadManager _uploadManager;
    private RightManager _rightManager;

    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _designsManager = (ModelDesignsManager) manager.lookup(ModelDesignsManager.ROLE);
        _i18nUtils = (I18nUtils) manager.lookup(org.ametys.core.util.I18nUtils.ROLE);
        _languageManager = (LanguagesManager) manager.lookup(LanguagesManager.ROLE);
        _modelsManager = (SkinModelsManager) manager.lookup(SkinModelsManager.ROLE);
        _skinFactoryManager = (SkinFactoryComponent) manager.lookup(SkinFactoryComponent.ROLE);
        _userManager = (UserManager) manager.lookup(UserManager.ROLE);
        _uploadManager = (UploadManager) manager.lookup(UploadManager.ROLE);
        _rightManager = (RightManager) manager.lookup(RightManager.ROLE);
    }
    
    @Override
    protected void checkUserRight(String skinName)
    {
        UserIdentity user = _userProvider.getUser();
        if (!_skinsManager.getSkinNameFromRequest().equals(skinName) || _rightManager.hasRight(user, "Plugins_SkinFactory_EditCurrentSkin", "/${WorkspaceName}") != RightResult.RIGHT_ALLOW)
        {
            throw new AccessDeniedException("User '" + user + "' tried perform operation on skin '" + skinName + "' without sufficient right");
        }
    }
    
    /**
     * Determines the skin directory is locked. If no, the lock owner is set in JSON map request attribute
     * @param skinDir The skin directory
     * @return information about the lock, or null if not locked
     * @throws IOException if an error occurs when manipulating files
     */
    protected Map<String, Object> checkLock(Path skinDir) throws IOException
    {
        if (!_lockManager.canWrite(skinDir))
        {
            Map<String, Object> result = new HashMap<>();

            UserIdentity lockOwner = _lockManager.getLockOwner(skinDir);
            User user = _userManager.getUser(lockOwner.getPopulationId(), lockOwner.getLogin());

            result.put("isLocked", true);
            result.put("lockOwner", user != null ? user.getFullName() + " (" + lockOwner + ")" : lockOwner);
            
            return result;
        }
        
        return null;
    }
    
    /**
     * Affect a new design configuration
     * @param skinName The skin name
     * @param designId The design id
     * @return the information on the design, or the error.
     * @throws IOException if an error occurs when manipulating files
     */
    @Callable(rights = "Plugins_SkinFactory_EditCurrentSkin", context = "/cms")
    public Map<String, Object> affectDesign(String skinName, String designId) throws IOException
    {
        Map<String, Object> result = new HashMap<>();
        
        Path tempDir = _skinHelper.getTempDirectory(skinName);
        String modelName = _skinHelper.getTempModel(skinName);
        
        Map<String, Object> lockInfos = checkLock(tempDir);
        if (lockInfos != null)
        {
            return lockInfos;
        }
        
        if (_modelsManager.getModel(modelName) == null)
        {
            result.put("unknownModel", true);
            result.put("modelName", modelName);
            return result;
        }
        
        _designsManager.applyDesign(modelName, designId, tempDir);
        
        Map<String, Object> values = new HashMap<>(_skinFactoryManager.getParameterValues(tempDir, modelName));
        result.put("parameters", values);
        
        String colorTheme = _skinFactoryManager.getColorTheme(tempDir);
        if (colorTheme != null)
        {
            result.put("themeId", colorTheme);
            result.put("colors", _modelsManager.getModel(modelName).getColors(colorTheme));
        }
        
        result.put("designId", designId);
        
        return result;
    }

    /**
     * Change the model of a skin
     * @param modelName The model name
     * @param skinName The skin name
     * @param useDefaults <code>true</code> to use default model parameters
     * @return The skin parameters, or the error informations.
     * @throws IOException if an error occurs when manipulating files
     * @throws TransformerConfigurationException if something goes wrong when generating the model file
     * @throws SAXException if an error occurs while saxing
     */
    @Callable(rights = "Plugins_SkinFactory_EditCurrentSkin", context = "/cms")
    public Map<String, Object> changeModel(String modelName, String skinName, boolean useDefaults) throws IOException, TransformerConfigurationException, SAXException
    {
        Path tempDir = _skinHelper.getTempDirectory(skinName);
        
        Map<String, Object> lockInfos = checkLock(tempDir);
        if (lockInfos != null)
        {
            return lockInfos;
        }
        
        String currentTheme = _skinFactoryManager.getColorTheme(tempDir);
        
        // Prepare skin in temporary file
        Path tmpDir = tempDir.getParent().resolve(skinName + "." + _DATE_FORMAT.format(new Date()));
        
        // Copy new model
        Path modelDir = _modelsManager.getModel(modelName).getPath();
        PathUtils.copyDirectory(modelDir, tmpDir, f -> !f.getFileName().equals("model"), false);
        
        // Apply all parameters
        _modelsManager.generateModelFile(tmpDir, modelName, currentTheme);
        
        if (useDefaults)
        {
            String defaultColorTheme = _modelsManager.getModel(modelName).getDefaultColorTheme();
            _skinFactoryManager.saveColorTheme(tmpDir, defaultColorTheme);
            _skinFactoryManager.applyModelParameters(modelName, tmpDir);
        }
        else
        {
            Map<String, Object> currentValues = _skinFactoryManager.getParameterValues(tempDir, modelName);
            _skinFactoryManager.applyModelParameters(modelName, tmpDir, currentValues);
        }
        
        _skinHelper.deleteQuicklyDirectory(tempDir);
        PathUtils.moveDirectory(tmpDir, tempDir);
        
        // Invalidate i18n.
        _skinHelper.invalidateTempSkinCatalogues(skinName);
        
        // Update lock file
        _lockManager.updateLockFile(tempDir, SKIN_FACTORY_TOOL_ID);
        
        return new HashMap<>(_skinFactoryManager.getParameterValues(tempDir, modelName));
    }
    
    /**
     * Get the languages available on a site
     * @param siteName The site name
     * @return The languages informations
     */
    @Callable(rights = Callable.NO_CHECK_REQUIRED)
    public Map<String, Object> getLanguages(String siteName)
    {
        Map<String, Object> languages = new LinkedHashMap<>();
        
        Site site = _siteManager.getSite(siteName);
        Skin skin = _skinsManager.getSkin(site.getSkinId());
        Path i18nDir = skin.getRawPath().resolve("i18n");
        
        Map<String, Language> allLanguages = _languageManager.getAvailableLanguages();
        
        try (Stream<Path> files = Files.list(i18nDir))
        {
            files.forEach(file ->
            {
                String fileName = file.getFileName().toString();
                if (!Files.isDirectory(file) && fileName.startsWith("messages"))
                {
                    String lang = null;
                    if (fileName.equals("messages.xml"))
                    {
                        lang = _getDefaultLanguage(file);
                    }
                    else
                    {
                        lang = fileName.substring("messages_".length(), fileName.lastIndexOf("."));
                    }
                    
                    if (allLanguages.containsKey(lang))
                    {
                        Language language = allLanguages.get(lang);
                        
                        Map<String, Object> langParams = new HashMap<>();
                        langParams.put("label", language.getLabel());
                        langParams.put("iconSmall", language.getSmallIcon());
                        langParams.put("iconMedium", language.getMediumIcon());
                        langParams.put("iconLarge", language.getLargeIcon());
                        
                        languages.put(lang, langParams);
                    }
                }
            });
        }
        catch (IOException e)
        {
            getLogger().error("Cannot get languages for site " + siteName, e);
        }
        
        return languages;
    }

    private String _getDefaultLanguage (Path i18nFile)
    {
        try (InputStream is = Files.newInputStream(i18nFile))
        {
            String string = org.apache.commons.io.IOUtils.toString(is, "UTF-8");

            // Not very pretty but more efficient than SAXparsing the all file to get the language
            int i = string.indexOf("xml:lang=\"");
            if (i != -1)
            {
                return string.substring(i + "xml:lang=\"".length(), i + "xml:lang=\"".length() + 2);
            }
            return null;
        }
        catch (IOException e)
        {
            throw new SkinParameterException ("Unable to parse file '" + i18nFile.getFileName().toString() + "'", e);
        }
    }
    
    /**
     * Get the colors of a model and its theme for a site.
     * @param siteName The site name
     * @return The colors and theme informations.
     */
    @Callable(rights = Callable.NO_CHECK_REQUIRED)
    public Map<String, Object> getColors(String siteName)
    {
        Map<String, Object> params = new LinkedHashMap<>();
        
        Site site = _siteManager.getSite(siteName);
        String skinId = site.getSkinId();
        String modelName = _skinHelper.getTempModel(skinId);
        
        Path tempDir = _skinHelper.getTempDirectory(skinId);
        String colorTheme = _skinFactoryManager.getColorTheme(tempDir);
        
        SkinModel model = _modelsManager.getModel(modelName);
        List<String> defaultColors = model.getDefaultColors();
        params.put("colors", defaultColors);
        
        if (StringUtils.isNotEmpty(colorTheme))
        {
            Theme theme = model.getTheme(colorTheme);
            if (theme != null)
            {
                params.put("themeId", theme.getId());
                params.put("themeColors", theme.getColors());
            }
        }
        
        return params;
    }
    
    /**
     * Get the css style items used by a site
     * @param siteName The site name
     * @return The css style items.
     */
    @Callable(rights = Callable.NO_CHECK_REQUIRED)
    public Map<String, Object> getCssStyleItems(String siteName)
    {
        Map<String, Object> styles = new LinkedHashMap<>();
        
        Site site = _siteManager.getSite(siteName);
        String skinId = site.getSkinId();
        String modelName = _skinHelper.getTempModel(skinId);
        
        Map<String, List<CssMenuItem>> styleItems = _modelsManager.getModel(modelName).getStyleItems();
        
        for (String styleId : styleItems.keySet())
        {
            List<Object> menuItems = new ArrayList<>();
            
            List<CssMenuItem> items = styleItems.get(styleId);
            for (CssMenuItem item : items)
            {
                if (item instanceof CssStyleItem)
                {
                    CssStyleItem cssItem = (CssStyleItem) item;
                    Map<String, String> itemParams = new HashMap<>();
                    
                    itemParams.put("value", cssItem.getValue());
                    itemParams.put("label", _i18nUtils.translate(cssItem.getLabel()));
                    
                    String iconCls = cssItem.getIconCls();
                    if (iconCls != null)
                    {
                        itemParams.put("iconCls", iconCls);
                    }
                    
                    String icon = cssItem.getIcon();
                    if (icon != null)
                    {
                        itemParams.put("icon", icon);
                    }
                    
                    String cssClass = cssItem.getCssClass();
                    if (cssClass != null)
                    {
                        itemParams.put("cssclass", cssClass);
                    }
                    
                    menuItems.add(itemParams);
                }
                else if (item instanceof Separator)
                {
                    menuItems.add("separator");
                }
            }
            
            styles.put(styleId, menuItems);
        }
        
        return styles;
    }
    
    /**
     * Get the parameters of the skin of a site
     * @param siteName The site name
     * @param paramIds If not null, specify the ids of the parameters to retrieve
     * @return The parameters
     */
    @Callable(rights = Callable.NO_CHECK_REQUIRED)
    public Map<String, Object> getParametersValues(String siteName, List<String> paramIds)
    {
        Map<String, Object> values = new LinkedHashMap<>();
        
        Site site = _siteManager.getSite(siteName);
        String skinId = site.getSkinId();
        String modelName = _skinHelper.getTempModel(skinId);
        
        Path tempDir = _skinHelper.getTempDirectory(skinId);
        
        if (paramIds != null)
        {
            values.putAll(_skinFactoryManager.getParameterValues(tempDir, modelName, paramIds));
        }
        else
        {
            values.putAll(_skinFactoryManager.getParameterValues(tempDir, modelName));
        }
        
        Map<String, Object> result = new LinkedHashMap<>();
        result.put("skinName", skinId);
        result.put("modelName", modelName);
        result.put("siteName", siteName);
        result.put("values", values);
        
        return result;
    }

    /**
     * Open the skin of a site for edition
     * @param siteName The site name
     * @param mode The open mode
     * @return The skin id, or an error message.
     * @throws IOException if an error occurs when manipulating files
     */
    @Callable(rights = "Plugins_SkinFactory_EditCurrentSkin", context = "/cms")
    public Map<String, String> openSkin(String siteName, String mode) throws IOException
    {
        Map<String, String> result = new HashMap<>();
        
        Site site = _siteManager.getSite(siteName);
        String skinId = site.getSkinId();
        
        Skin skin = _skinsManager.getSkin(skinId);
        if (!skin.isModifiable())
        {
            throw new IllegalStateException("The skin '" + skinId + "' is not modifiable and thus cannot be opened in skin editor.");
        }
        
        Path tempDir = _skinHelper.getTempDirectory(skinId);
        Path workDir = _skinHelper.getWorkDirectory(skinId);
        Path skinDir = _skinHelper.getSkinDirectory(skinId);
        
        String modelName = null;
        if (__PROD_MODE.equals(mode))
        {
            modelName = _skinHelper.getSkinModel(skinId);
        }
        else if (__WORK_MODE.equals(mode))
        {
            modelName = _skinHelper.getWorkModel(skinId);
        }
        else
        {
            modelName = _skinHelper.getTempModel(skinId);
        }
        
        SkinModel model = _modelsManager.getModel(modelName);
        if (model == null)
        {
            result.put("model-not-found", "true");
            return result;
        }
        
        String modelHash = _modelsManager.getModelHash(modelName);
        
        if (__PROD_MODE.equals(mode) || __WORK_MODE.equals(mode))
        {
            // Delete temp directory if exists
            if (Files.exists(tempDir))
            {
                _skinHelper.deleteQuicklyDirectory(tempDir);
            }
            
            if (__PROD_MODE.equals(mode))
            {
                // Delete work directory if exists
                if (Files.exists(workDir))
                {
                    _skinHelper.deleteQuicklyDirectory(workDir);
                }
                
                // Copy from skin
                PathUtils.copyDirectory(skinDir, workDir, FileFilter.getSkinFileFilter());
            }
                    
            boolean isUpTodate = modelHash.equals(_getHash (workDir));
            if (!isUpTodate)
            {
                // Re-apply model to work directory
                _reapplyModel(workDir, model.getPath(), modelHash);
            }
            
            // Apply parameters
            _skinFactoryManager.applyModelParameters(modelName, workDir);
            
            // Copy work in temp
            PathUtils.copyDirectory(workDir, tempDir);
            
            // Create .lock file
            _lockManager.updateLockFile(tempDir, SKIN_FACTORY_TOOL_ID);
        }
        else
        {
            boolean isUpTodate = modelHash.equals(_getHash (tempDir));
            if (!isUpTodate)
            {
                // Re-apply model to temp directory
                _reapplyModel(tempDir, model.getPath(), modelHash);
            }
            
            // Apply parameters
            _skinFactoryManager.applyModelParameters(modelName, tempDir);
            
            // Update .lock file
            _lockManager.updateLockFile(tempDir, SKIN_FACTORY_TOOL_ID);
        }
        
        result.put("skinId", skinId);
        return result;
    }
    
    private void _reapplyModel (Path skinDir, Path modelDir, String hash) throws IOException
    {
        // Make a copy of model.xml file
        Path xmlFile = skinDir.resolve("model.xml");
        _preserveFile (skinDir, xmlFile);
        
        // Preserve uploaded images
        Path uploadDir = skinDir.resolve("model/_uploads");
        if (Files.exists(uploadDir))
        {
            _preserveFile(skinDir, uploadDir.getParent());
        }
        
        // Delete old directory
        PathUtils.deleteQuietly(skinDir);
        
        // Copy the model
        PathUtils.copyDirectory(modelDir, skinDir, FileFilter.getModelFilter(modelDir), false);
        
        // Copy files to preserve
        _copyFilesToPreserve (skinDir);
        
        // Update hash
        _skinFactoryManager.updateHash(xmlFile, hash);
    }
    
    private void _preserveFile(Path skinDir, Path fileToPreserve) throws IOException
    {
        Path toPreserveDir = skinDir.getParent().resolve(skinDir.getFileName().toString() + "_tmp");
        if (Files.isDirectory(fileToPreserve))
        {
            PathUtils.moveDirectoryToDirectory(fileToPreserve, toPreserveDir, true);
        }
        else
        {
            PathUtils.moveFileToDirectory(fileToPreserve, toPreserveDir, true);
        }
        
    }
    
    private void _copyFilesToPreserve(Path skinDir) throws IOException
    {
        Path toPreserveDir = skinDir.getParent().resolve(skinDir.getFileName().toString() + "_tmp");
        if (Files.exists(toPreserveDir))
        {
            try (Stream<Path> children = Files.list(toPreserveDir))
            {
                children.forEach(LambdaUtils.wrapConsumer(child ->
                {
                    if (Files.isDirectory(child))
                    {
                        PathUtils.moveDirectoryToDirectory(child, skinDir, false);
                    }
                    else
                    {
                        PathUtils.moveFileToDirectory(child, skinDir, false);
                    }
                }));
            }
            catch (LambdaException e)
            {
                throw (IOException) e.getCause();
            }
            
            PathUtils.deleteQuietly(toPreserveDir);
        }
    }
    
    private String _getHash(Path skinDir)
    {
        Path modelFile = skinDir.resolve("model.xml");
        if (!Files.exists(modelFile))
        {
            // No model
            return null;
        }
        
        try (InputStream is = Files.newInputStream(modelFile))
        {
            XPath xpath = XPathFactory.newInstance().newXPath();
            return xpath.evaluate("model/@hash", new InputSource(is));
        }
        catch (XPathExpressionException e)
        {
            throw new IllegalStateException("The id of model is missing", e);
        }
        catch (IOException e)
        {
            getLogger().error("Can not determine the hash the skin", e);
            return null;
        }
    }
    
    /**
     * Restore the default parameters for a skin
     * @param skinName The skin name
     * @return The skin informations, or an error code.
     * @throws IOException if an error occurs when manipulating files
     * @throws TransformerConfigurationException if something goes wrong when generating the model file
     * @throws SAXException if an error occurs while saxing
     */
    @Callable(rights = "Plugins_SkinFactory_EditCurrentSkin", context = "/cms")
    public Map<String, Object> restoreDefaults(String skinName) throws IOException, TransformerConfigurationException, SAXException
    {
        Map<String, Object> result = new HashMap<>();
        
        Path tempDir = _skinHelper.getTempDirectory(skinName);
        String modelName = _skinHelper.getTempModel(skinName);
        
        Map<String, Object> lockInfos = checkLock(tempDir);
        if (lockInfos != null)
        {
            return lockInfos;
        }
        
        if (_modelsManager.getModel(modelName) == null)
        {
            result.put("unknownModel", true);
            result.put("modelName", modelName);
            return result;
        }
        
        // Prepare skin in temporary file
        Path tmpDir = tempDir.getParent().resolve(skinName + "." + _DATE_FORMAT.format(new Date()));
        
        // Copy new model
        SkinModel model = _modelsManager.getModel(modelName);
        Path modelDir = model.getPath();
        PathUtils.copyDirectory(modelDir, tmpDir, f -> !f.getFileName().toString().equals("model"), false);
        
        
        _modelsManager.generateModelFile(tmpDir, modelName);
        
        String defaultColorTheme = model.getDefaultColorTheme();
        if (defaultColorTheme != null)
        {
            // Save color theme
            _skinFactoryManager.saveColorTheme(tmpDir, defaultColorTheme);
            
            result.put("themeId", defaultColorTheme);
            result.put("colors", model.getColors(defaultColorTheme));
        }
        
        // Apply all parameters
        _skinFactoryManager.applyModelParameters(modelName, tmpDir);
        
        _skinHelper.deleteQuicklyDirectory(tempDir);
        PathUtils.moveDirectory(tmpDir, tempDir);
        
        // Invalidate i18n.
        _skinHelper.invalidateTempSkinCatalogues(skinName);
        
        // Update lock file
        _lockManager.updateLockFile(tempDir, SKIN_FACTORY_TOOL_ID);
        
        Map<String, Object> values = new HashMap<>(_skinFactoryManager.getParameterValues(tempDir, modelName));
        result.put("parameters", values);
        
        return result;
    }

    /**
     * Set the theme used by a skin
     * @param skinName The skin name
     * @param themeId The theme id
     * @return The theme informations, or an error code.
     * @throws IOException if an error occurs when manipulating files
     */
    @Callable(rights = "Plugins_SkinFactory_EditCurrentSkin", context = "/cms")
    public Map<String, Object> updateColorTheme(String skinName, String themeId) throws IOException
    {
        Map<String, Object> result = new HashMap<>();

        Path tempDir = _skinHelper.getTempDirectory(skinName);
        String modelName = _skinHelper.getTempModel(skinName);
        
        Map<String, Object> lockInfos = checkLock(tempDir);
        if (lockInfos != null)
        {
            return lockInfos;
        }
        
        if (_modelsManager.getModel(modelName) == null)
        {
            result.put("unknownModel", true);
            result.put("modelName", modelName);
            return result;
        }
        
        // Save color theme
        _skinFactoryManager.saveColorTheme(tempDir, themeId);
        
        // Apply new color theme
        _skinFactoryManager.applyColorTheme (modelName, tempDir);
        
        // Update lock
        _lockManager.updateLockFile(tempDir, SKIN_FACTORY_TOOL_ID);
        
        result.put("themeId", themeId);
        result.put("colors", _modelsManager.getModel(modelName).getColors(themeId));
        
        return result;
    }

    /**
     * Update a parameter of the skin
     * @param skinName The skin name
     * @param lang The current language
     * @param parameterId The parameter id to update
     * @param value The new value for the parameter
     * @param uploaded <code>true</code> if the file was uploaded
     * @return The skin parameters updated, or an error code.
     * @throws IOException if an error occurs when manipulating files
     */
    @Callable(rights = "Plugins_SkinFactory_EditCurrentSkin", context = "/cms")
    public Map<String, Object> updateParameter(String skinName, String lang, String parameterId, String value, boolean uploaded) throws IOException
    {
        Path tempDir = _skinHelper.getTempDirectory(skinName);
        
        Map<String, Object> lockInfos = checkLock(tempDir);
        if (lockInfos != null)
        {
            return lockInfos;
        }
        
        String modelName = _skinHelper.getTempModel(skinName);
        if (modelName == null)
        {
            Map<String, Object> result = new HashMap<>();

            result.put("unknownModel", true);
            result.put("modelName", modelName);
            return result;
        }
        
        Map<String, AbstractSkinParameter> modelParameters = _skinFactoryManager.getModelParameters(modelName);
        AbstractSkinParameter skinParameter = modelParameters.get(parameterId);
        if (skinParameter != null)
        {
            // Apply parameter
            if (skinParameter instanceof ImageParameter)
            {
                FileValue fileValue = new FileValue(value, uploaded);
                _skinFactoryManager.applyParameter(skinParameter, tempDir, modelName, fileValue, lang);
            }
            else
            {
                _skinFactoryManager.applyParameter(skinParameter, tempDir, modelName, value, lang);
            }
            
            // Save parameter
            if (skinParameter instanceof I18nizableTextParameter)
            {
                Map<String, String> values = new HashMap<>();
                values.put(lang, value);
                _skinFactoryManager.saveParameter(tempDir, parameterId, values);
                
                _skinHelper.invalidateTempSkinCatalogue (skinName, lang);
            }
            else if (skinParameter instanceof ImageParameter)
            {
                FileValue fileValue = new FileValue(value, uploaded);
                _skinFactoryManager.saveParameter(tempDir, parameterId, fileValue);
            }
            else
            {
                _skinFactoryManager.saveParameter(tempDir, parameterId, value);
            }
           
            
            // Update lock
            _lockManager.updateLockFile(tempDir, SKIN_FACTORY_TOOL_ID);
        }
        
        return new HashMap<>(_skinFactoryManager.getParameterValues(tempDir, modelName));
    }
    
    /**
     * Upload a local image and set it as value for a image parameter
     * @param uploadId The upload identifier
     * @param fileName The name of uploaded file
     * @param skinName The skin name
     * @param parameterId The parameter id to update
     * @return The skin parameters updated, or an error code.
     * @throws IOException if an error occurs when manipulating files
     */
    @Callable(rights = "Plugins_SkinFactory_EditCurrentSkin", context = "/cms")
    public Map<String, Object> uploadLocalImage (String uploadId, String fileName, String skinName, String parameterId) throws IOException
    {
        Path tempDir = _skinHelper.getTempDirectory(skinName);
        
        Map<String, Object> lockInfos = checkLock(tempDir);
        if (lockInfos != null)
        {
            return lockInfos;
        }
        
        String modelName = _skinHelper.getTempModel(skinName);
        if (modelName == null)
        {
            Map<String, Object> result = new HashMap<>();

            result.put("unknownModel", true);
            result.put("modelName", modelName);
            return result;
        }
        
        ImageParameter imgParam = (ImageParameter) _skinFactoryManager.getModelParamater(modelName, parameterId);
        
        Upload upload = null;
        try
        {
            upload = _uploadManager.getUpload(_userProvider.getUser(), uploadId);
            
            // Copy uploaded file into model
            Path uploadDir = _getUploadDir (tempDir, imgParam);
            Path uploadFile = uploadDir.resolve(fileName);
            
            try (OutputStream os = Files.newOutputStream(uploadFile);
                 InputStream is = upload.getInputStream())
            {
                IOUtils.copy(is, os);
            }
            catch (IOException e)
            {
                // close quietly
            }
        }
        catch (NoSuchElementException e)
        {
            // Invalid upload id
            getLogger().error(String.format("Cannot find the temporary uploaded file for id '%s' and login '%s'.", uploadId, _userProvider.getUser()), e);

            Map<String, Object> result = new HashMap<>();
            result.put("uploadFailed", true);
            result.put("uploadId", uploadId);
            return result;
        }
        
        FileValue fileValue = new ImageParameter.FileValue(fileName, true);
        
        // Apply parameter
        _skinFactoryManager.applyParameter(imgParam, tempDir, modelName, fileValue, null);
        
        // Save parameter
        _skinFactoryManager.saveParameter(tempDir, imgParam.getId(), fileValue);
        
        // Update lock
        _lockManager.updateLockFile(tempDir, SKIN_FACTORY_TOOL_ID);
        
        return new HashMap<>(_skinFactoryManager.getParameterValues(tempDir, modelName));
    }

    /**
     * Retrieve the list of images for the skin and parameter
     * @param skinName The skin name
     * @param paramId The parameter id
     * @return The map of images informations
     * @throws IOException if an error occurs when manipulating files
     */
    @Callable(rights = "Plugins_SkinFactory_EditCurrentSkin", context = "/cms")
    public Map<String, Object> getGalleryImages(String skinName, String paramId) throws IOException
    {
        Map<String, Object> gallery = new HashMap<>();
        
        String modelName = _skinHelper.getTempModel(skinName);
        SkinModel model = _modelsManager.getModel(modelName);
     
        if (model != null)
        {
            AbstractSkinParameter skinParameter = _skinFactoryManager.getModelParamater(modelName, paramId);
            if (skinParameter instanceof ImageParameter)
            {
                ImageParameter imageParam = (ImageParameter) skinParameter;
                
                Path imageDir = model.getPath().resolve("model/images");
                Path libraryFile = imageDir.resolve(imageParam.getLibraryPath());
                gallery.put("gallery", _imageFiles2JsonObject(imageDir.toAbsolutePath().toString(), libraryFile.toAbsolutePath().toString(), libraryFile, modelName, true));
                
                // Uploaded local images
                Path tempDir = _skinHelper.getTempDirectory(skinName);
                Path uploadDir = tempDir.resolve("model/_uploads/" + imageParam.getLibraryPath());
                if (Files.exists(uploadDir))
                {
                    gallery.put("uploadedGroup", _uploadImages2JsonObject(uploadDir, skinName, imageParam));
                }
            }
        }
        else
        {
            getLogger().warn("Unable to get gallery images : the model '" + modelName + "' does not exist anymore");
        }

        return gallery;
    }
    
    private Map<String, Object> _uploadImages2JsonObject(Path uploadDir, String skinName, ImageParameter imageParam) throws IOException
    {
        Map<String, Object> uploadedGroup = new HashMap<>();
        
        uploadedGroup.put("label", new I18nizableText("plugin.skinfactory", "PLUGINS_SKINFACTORY_IMAGESGALLERY_GROUP_UPLOADED"));
        
        List<Object> uploadedImages = new ArrayList<>();
        
        try (Stream<Path> files = Files.list(uploadDir))
        {
            files
                .filter(f -> _isImage(f))
                .forEach(child ->
                {
                    Map<String, Object> jsonObject = new HashMap<>();
                    jsonObject.put("type", "image");
                    jsonObject.put("filename", child.getFileName().toString());
                    jsonObject.put("src", child.getFileName().toString());
                    jsonObject.put("thumbnail", "/plugins/skinfactory/" + skinName + "/_thumbnail/64/64/model/_uploads/" + (imageParam.getLibraryPath() + '/' + child.getFileName().toString()).replaceAll("\\\\", "/"));
                    jsonObject.put("thumbnailLarge", "/plugins/skinfactory/" + skinName + "/_thumbnail/100/100/model/_uploads/" + (imageParam.getLibraryPath() + '/' + child.getFileName().toString()).replaceAll("\\\\", "/"));
                    jsonObject.put("uploaded", true);
                    uploadedImages.add(jsonObject);
                });
        }
        uploadedGroup.put("images", uploadedImages);
        
        return uploadedGroup;
    }

    private List<Object> _imageFiles2JsonObject(String imageDirPath, String libraryDirPath, Path file, String modelName, boolean deep) throws IOException
    {
        List<Object> imageFilesJsonObject = new ArrayList<>();
        
        try (Stream<Path> children = Files.list(file))
        {
            children
                .forEach(LambdaUtils.wrapConsumer(child ->
                {
                    Map<String, Object> jsonObject = new HashMap<>();

                    if (Files.isDirectory(child) && deep && !child.getFileName().equals(".svn"))
                    {
                        jsonObject.put("type", "group");
                        jsonObject.put("label", child.getFileName().toString());
                        jsonObject.put("childs", _imageFiles2JsonObject(imageDirPath, libraryDirPath, child, modelName, false));
                         
                        imageFilesJsonObject.add(jsonObject);
                    }
                    else if (_isImage(child))
                    {
                        jsonObject.put("type", "image");
                        jsonObject.put("filename", child.getFileName().toString());
                        jsonObject.put("src", child.toAbsolutePath().toString().substring(libraryDirPath.length() + 1).replaceAll("\\\\", "/"));
                        jsonObject.put("thumbnail", "/plugins/skinfactory/" + modelName + "/_thumbnail/64/64/model/images/" + child.toAbsolutePath().toString().substring(imageDirPath.length() + 1).replaceAll("\\\\", "/"));
                        jsonObject.put("thumbnailLarge", "/plugins/skinfactory/" + modelName + "/_thumbnail/100/100/model/images/" + child.toAbsolutePath().toString().substring(imageDirPath.length() + 1).replaceAll("\\\\", "/"));
                         
                        imageFilesJsonObject.add(jsonObject);
                    }
                }));
        }
        catch (LambdaException e)
        {
            throw (IOException) e.getCause();
        }
        
        return imageFilesJsonObject;
    }
    
    private Path _getUploadDir (Path tempDir, ImageParameter imgParam) throws IOException
    {
        Path uploadDir = tempDir.resolve("model/_uploads/" + imgParam.getLibraryPath());
        if (!Files.exists(uploadDir))
        {
            Files.createDirectories(uploadDir);
        }
        return uploadDir;
    }

    private boolean _isImage(Path file)
    {
        String name = file.getFileName().toString().toLowerCase();
        int index = name.lastIndexOf(".");
        String ext = name.substring(index + 1);
        
        if (name.equals("thumbnail_16.png") || name.equals("thumbnail_32.png") || name.equals("thumbnail_48.png"))
        {
            return false;
        }

        return "png".equals(ext) || "gif".equals(ext) || "jpg".equals(ext) || "jpeg".equals(ext);
    }

    /**
     * Retrieve the list of gallery variants available for the specified skin and parameter
     * @param skinName The skin name
     * @param paramId The parameter id
     * @return The list of gallery variants
     * @throws IOException if an error occurs when manipulating files
     */
    @Callable(rights = "Plugins_SkinFactory_EditCurrentSkin", context = "/cms")
    public List<Object> getGalleryVariants(String skinName, String paramId) throws IOException
    {
        List<Object> galleryVariants = new ArrayList<>();
        
        String modelName = _skinHelper.getTempModel(skinName);
     
        AbstractSkinParameter skinParameter = _skinFactoryManager.getModelParamater(modelName, paramId);
        if (skinParameter instanceof VariantParameter)
        {
            VariantParameter variantParam = (VariantParameter) skinParameter;
            
            List<Variant> variants = variantParam.getVariants();
            for (Variant variant : variants)
            {
                Map<String, Object> jsonObject = new HashMap<>();
                
                jsonObject.put("value", variant.getId());
                
                String thumbnail = variant.getThumbnail();
                if (thumbnail != null)
                {
                    jsonObject.put("thumbnail", "/plugins/skinfactory/" + modelName + "/_thumbnail/32/32/model/variants/" + thumbnail);
                }
                else
                {
                    jsonObject.put("thumbnail", "/plugins/skinfactory/resources/img/variant_default_32.png");
                }
                
                jsonObject.put("label", variant.getLabel());
                jsonObject.put("description", variant.getDescription());
                
                galleryVariants.add(jsonObject);
            }
        }

        return galleryVariants;
    }
    
    /**
     * Retrieve the list of themes' colors for a site
     * @param siteName The site name
     * @return The model's themes colors
     */
    @Callable(rights = Callable.NO_CHECK_REQUIRED)
    public List<Object> getThemeColors(String siteName)
    {
        List<Object> themesJsonObject = new ArrayList<>();
        
        String skinId = _siteManager.getSite(siteName).getSkinId();
        String modelName = _skinHelper.getTempModel(skinId);
        
        SkinModel model = _modelsManager.getModel(modelName);
        if (model != null)
        {
            Map<String, Theme> themes = model.getThemes();
            for (String name : themes.keySet())
            {
                Map<String, Object> jsonObject = new HashMap<>();
                
                Theme theme = themes.get(name);
                
                jsonObject.put("id", theme.getId());
                jsonObject.put("label", theme.getLabel());
                jsonObject.put("colors", theme.getColors());

                themesJsonObject.add(jsonObject);
            }
        }
        else
        {
            getLogger().warn("Unable to get theme colors : the model '" + modelName + "' does not exist anymore");
        }
        
        return themesJsonObject;
    }
}
