/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.skinfactory.readers;

import java.awt.Dimension;
import java.awt.image.BufferedImage;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.Serializable;
import java.nio.file.Path;
import java.util.Map;

import javax.imageio.ImageIO;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.caching.CacheableProcessingComponent;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Response;
import org.apache.cocoon.environment.SourceResolver;
import org.apache.cocoon.reading.ServiceableReader;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.excalibur.source.SourceValidity;
import org.xml.sax.SAXException;

import org.ametys.core.util.ImageHelper;
import org.ametys.core.util.URIUtils;
import org.ametys.core.util.path.PathSource;
import org.ametys.plugins.skincommons.SkinEditionHelper;
import org.ametys.web.skin.SkinModel;
import org.ametys.web.skin.SkinModelsManager;

import net.coobird.thumbnailator.makers.FixedSizeThumbnailMaker;
import net.coobird.thumbnailator.resizers.DefaultResizerFactory;

/**
 * Reader for resource of the skin
 */
public class SkinResourceReader extends ServiceableReader implements CacheableProcessingComponent
{
    private SkinModelsManager _modelsManager;
    private SkinEditionHelper _skinHelper;
    
    private PathSource _source;
    
    private int _width;
    private int _height;
    private int _maxWidth;
    private int _maxHeight;
    
    @Override
    public void service(ServiceManager sManager) throws ServiceException
    {
        super.service(sManager);
        _modelsManager = (SkinModelsManager) sManager.lookup(SkinModelsManager.ROLE);
        _skinHelper = (SkinEditionHelper) sManager.lookup(SkinEditionHelper.ROLE);
    }
    
    @Override
    public void setup(SourceResolver sResolver, Map objModel, String src, Parameters par) throws ProcessingException, SAXException, IOException
    {
        super.setup(sResolver, objModel, src, par);
        
        // parameters for image resizing
        _width = par.getParameterAsInteger("width", 0);
        _height = par.getParameterAsInteger("height", 0);
        _maxWidth = par.getParameterAsInteger("maxWidth", 0);
        _maxHeight = par.getParameterAsInteger("maxHeight", 0);
        
        String path = par.getParameter("path", null);
        assert path != null;
        
        Path rootDir = null;
        String modelName = par.getParameter("modelName", null);
        if (StringUtils.isNotEmpty(modelName))
        {
            SkinModel model = _modelsManager.getModel(modelName);
            rootDir = model.getPath();
            _source = new PathSource("model", "model:" + modelName + "://" + path, rootDir.resolve(URIUtils.decode(path)));
        }
        else
        {
            String skinName = par.getParameter("skinName", null);
            rootDir = _skinHelper.getTempDirectory(skinName);
            _source = new PathSource("file", rootDir.resolve(URIUtils.decode(path)));
        }
    }
    
    @Override
    public Serializable getKey()
    {
        return _source.getFile().toAbsolutePath().toString() + "#" + _height + "#" + _width + "#" + _maxHeight + "#" + _maxWidth;
    } 

    @Override
    public SourceValidity getValidity()
    {
        return _source.getValidity();
    }
    
    @Override
    public long getLastModified()
    {
        return _source.getLastModified();
    }
    
    @Override
    public String getMimeType()
    {
        return _source.getMimeType();
    }
    
    @Override
    public void generate() throws IOException, SAXException, ProcessingException
    {
        String name = _source.getName();
        name = name.replaceAll("\\\\", "\\\\\\\\");
        name = name.replaceAll("\\\"", "\\\\\\\"");
        
        Response response = ObjectModelHelper.getResponse(objectModel);
        
        try (InputStream is  = _source.getInputStream())
        {
            if (_width > 0 || _height > 0)
            {
                // it's an image, which must be resized
                int i = name.lastIndexOf('.');
                String format = name.substring(i + 1);
                
                ImageHelper.generateThumbnail(is, out, format, _height, _width, 0, 0);
            }
            else if (_maxHeight > 0 || _maxWidth > 0)
            {
                // it's an image, which must be resized
                int i = name.lastIndexOf('.');
                String format = name.substring(i + 1);
                
                byte[] fileContent = IOUtils.toByteArray(is);
                BufferedImage src = ImageHelper.read(new ByteArrayInputStream(fileContent));
                
                _generateThumbnail (out, format, src, fileContent, _maxHeight, _maxWidth);
            }
            else
            {
                response.setHeader("Content-Length", Long.toString(_source.getContentLength()));
                IOUtils.copy(is, out);
            }
            
            out.flush();
        }
        catch (Exception e)
        {
            throw new ProcessingException("Unable to download file of uri " + _source.getURI(), e);
        }
    }
    
    @Override
    public void recycle()
    {
        super.recycle();
        _source = null;
    }
    
    private void _generateThumbnail (OutputStream os, String format, BufferedImage src, byte[] fileContent, int maxHeight, int maxWidth) throws IOException
    {
        int srcHeight = src.getHeight();
        int srcWidth = src.getWidth();
        
        if (maxWidth == srcWidth && maxHeight == srcHeight)
        {
            IOUtils.write(fileContent, os);
            return;
        }
        if (srcWidth < maxWidth && srcHeight < maxHeight)
        {
            // Image is too small : zoom them crop image to minHeight x minWidth dimension
            _generateZoomAndCropImage (out, format, src, fileContent, maxHeight, maxWidth);
        }
        else
        {
            BufferedImage dest = ImageHelper.generateThumbnail(src, _height, _width, _maxHeight, _maxWidth);
            if (src == dest)
            {
                // Thumbnail is equals to src image, means that the image is the same
                // We'd rather like return the initial stream
                IOUtils.write(fileContent, os);
            }
            else
            {
                ImageIO.write(dest, format, os);
            }
        }
    }
    
    private void _generateZoomAndCropImage (OutputStream os, String format,  BufferedImage src, byte[] fileContent, int minHeight, int minWidth) throws IOException
    {
        int srcHeight = src.getHeight();
        int srcWidth = src.getWidth();
        
        int destHeight = 0;
        int destWidth = 0;
        
        Dimension srcDimension = new Dimension(srcWidth, srcHeight);
        
        boolean keepAspectRatio = true;
        
        if (srcWidth > srcHeight)
        {
            destHeight = minHeight;
            
            // width is computed keeping ratio
            destWidth = srcWidth * destHeight / srcHeight;
        }
        else
        {
            destWidth = minWidth;
         
            // dest is computed keeping ratio
            destHeight = srcHeight * destWidth / srcWidth;
        }
        
        Dimension thumbnailDimension = new Dimension(destWidth, destHeight);
        
        
        BufferedImage thumbImage = new FixedSizeThumbnailMaker(destWidth, destHeight, keepAspectRatio, true)
                                   .resizer(DefaultResizerFactory.getInstance().getResizer(srcDimension, thumbnailDimension)) 
                                   .imageType(src.getColorModel().hasAlpha() ? BufferedImage.TYPE_INT_ARGB : BufferedImage.TYPE_INT_RGB)
                                   .make(src); 
        
        BufferedImage cropImage = _getCropImage (thumbImage, 0, 0, minHeight, minHeight);
        
        if (src == cropImage)
        {
            // Thumbnail is equals to src image, means that the image is the same
            // We'd rather like return the initial stream
            IOUtils.write(fileContent, os);
        }
        else
        {
            ImageIO.write(cropImage, format, os);
        }
    }
    
    private BufferedImage _getCropImage (BufferedImage src, int x, int y, int width, int height)
    {
        int srcHeight = src.getHeight();
        int srcWidth = src.getWidth();
        
        int w = width;
        if (width + x > srcWidth)
        {
            w = srcWidth - x;
        }
        
        int h = height;
        if (height + y > srcHeight)
        {
            h = srcHeight - y;
        }
        
        return src.getSubimage(x, y, w, h);
    }
}
