/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * @private
 * This helper is used to change the model of the current skin. See #open method.
 */
Ext.define("Ametys.plugins.skinfactory.helper.ChangeModelUI", {
    singleton: true,

    /**
     * @private
     * @property {Function} _cbFn Callback function to execute after the user validates the dialog
     */

    /**
     * @private
     * @property {Ametys.window.DialogBox} _box The dialog box
     */

    /**
     * @private
     * @property {Ext.form.FormPanel} _formPanel The form of the dialog box
     */

    /**
     * Open the dialog box for changing the model
     * @param {String} [modelId] The id of model to change to.
     * @param {String} [skinId] The id of skin being changed.
     * @param {Function} [callback] The function to call when successful
     */
    open: function (modelId, skinId, callback)
    {
        this._cbFn = callback;

        this._delayedInitialize();

        this._box.show();
        this._initForm (modelId, skinId);
    },

    /**
     * Creates the dialog box
     * @private
     */
    _delayedInitialize: function ()
    {
        if (this._initialized)
        {
            return;
        }

        this._formPanel = this._createFormPanel();

        this._box = Ext.create('Ametys.window.DialogBox', {
            iconCls: 'ametysicon-painter-palette',
            title: "{{i18n PLUGINS_SKINFACTORY_CHANGE_MODEL_TITLE}}",
            width: 580,
            scrollable: false,

            items: [this._formPanel],
            
            defaultFocus: 'useDefaults',
            closeAction: 'hide',
           
            referenceHolder: true,
            defaultButton: 'validate',
            
            buttons: [{
            	reference: 'validate',
                text: "{{i18n plugin.skincommons:PLUGINS_SKINCOMMONS_CANCELCHANGES_DIALOG_OK_BTN}}",
                handler: Ext.bind(this._validate, this)
            }, {
                text: "{{i18n plugin.skincommons:PLUGINS_SKINCOMMONS_CANCELCHANGES_DIALOG_CANCEL_BTN}}",
                handler: Ext.bind(this._cancel, this)
            }]
        });

        this._initialized = true;
    },

    /**
     * Create the form panel
     * @return {Ext.form.FormPanel} the form panel
     * @private
     */
    _createFormPanel: function()
    {
        var formPanel = new Ext.form.FormPanel( {
            defaultType: 'textfield',
            border: false,
            scrollable: true,

            defaults: {
                cls: 'ametys',
                labelAlign: 'right',
                labelSeparator: '',
                labelWidth: 80,
                width: 470,
                msgTarget: 'side'
            },

            items: [
                {
                    xtype: 'container',
                    html: "{{i18n PLUGINS_SKINFACTORY_CHANGE_MODEL_HINT}}",
                    cls: 'text'
                },
                {
                    xtype: "hidden",
                    name: "modelId"
                },
                {
                    xtype: "hidden",
                    name: "skinId"
                },
                {
                    xtype: 'radio',
                    boxLabel: "{{i18n PLUGINS_SKINFACTORY_CHANGE_MODEL_USE_DEFAULT_PARAMS}}",
                    name: "useDefaults",
                    itemId: 'useDefaults',
                    inputValue: true,
                    checked: true
                },
                {
                    xtype: 'radio',
                    boxLabel: "{{i18n PLUGINS_SKINFACTORY_CHANGE_MODEL_USE_CURRENT_PARAMS}}",
                    name: "useDefaults",
                    inputValue: false
                }
            ]
        });

        return formPanel;
    },

    /**
     * Initialize the form
     * @param {String} [modelId] The id of the new model for the skin.
     * @param {String} [skinId] The id of the skin being changed.
     * @private
     */
    _initForm: function (modelId, skinId)
    {
    	var form = this._formPanel.getForm();
        form.findField('modelId').setValue(modelId);
        form.findField('skinId').setValue(skinId);
    },

   /**
    * Handler for the 'Ok' button of the dialog box.
    * Generate a new skin from selected model.
    * @private
    */
    _validate: function ()
    {
        var form = this._formPanel.getForm();
        if (!form.isValid())
        {
            return;
        }

        var modelId = form.findField('modelId').getValue();
        var skinId = form.findField('skinId').getValue();
        var useDefaults = form.findField("useDefaults").getGroupValue();

        Ametys.plugins.skinfactory.skin.SkinDAO.changeModel([modelId, skinId, useDefaults], Ext.bind(this._changeModelCb, this), {
            waitMessage: {
                target: this._box
            }
        });
    },

    /**
     * Callback for the "cancel" button of the dialog. Close the dialog.
     * @private
     */
    _cancel: function ()
    {
        this._box.hide();
    },

    /**
     * Callback after the changing the model of the skin.
     * @param {Object} response The skin parameters, or the error informations.
     * @private
     */
    _changeModelCb: function (response)
    {
        this._box.hide();
        if (Ext.isFunction(this._cbFn))
        {
            this._cbFn (response);
        }
    }

});
