/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.skinfactory.actions;

import java.io.IOException;
import java.nio.file.Path;
import java.util.HashMap;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.acting.ServiceableAction;
import org.apache.cocoon.environment.Request;

import org.ametys.core.cocoon.JSonReader;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.User;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.user.UserManager;
import org.ametys.plugins.skincommons.SkinEditionHelper;
import org.ametys.plugins.skincommons.SkinLockManager;
import org.ametys.skinfactory.SkinFactoryComponent;
import org.ametys.web.repository.site.SiteManager;
import org.ametys.web.skin.SkinModelsManager;
import org.ametys.web.skin.SkinsManager;

/**
 * Abstract class for manipulating skin directory
 *
 */
public abstract class AbstractSkinAction extends ServiceableAction
{
    /** Constant for skin editor tool id */
    public static final String SKIN_FACTORY_TOOL_ID = "uitool-skinfactory";
    
    /** The cocoon context */
    protected org.apache.cocoon.environment.Context _cocoonContext;
    /** The user provider */
    protected CurrentUserProvider _userProvider;
    /** The sites manager */
    protected SiteManager _siteManager;
    /** The skin factory component */
    protected SkinFactoryComponent _skinFactoryManager;
    /** The models manager */
    protected SkinModelsManager _modelsManager;
    /** The lock manager */
    protected SkinLockManager _lockManager;
    /** The skin edition helper */
    protected SkinEditionHelper _skinHelper;
    /** The skin manager */
    protected SkinsManager _skinsManager;
    /** The users manager */
    protected UserManager _userManager;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _siteManager = (SiteManager) smanager.lookup(SiteManager.ROLE);
        _skinFactoryManager = (SkinFactoryComponent) smanager.lookup(SkinFactoryComponent.ROLE);
        _modelsManager = (SkinModelsManager) smanager.lookup(SkinModelsManager.ROLE);
        _lockManager = (SkinLockManager) smanager.lookup(SkinLockManager.ROLE);
        _skinHelper = (SkinEditionHelper) smanager.lookup(SkinEditionHelper.ROLE);
        _skinsManager = (SkinsManager) smanager.lookup(SkinsManager.ROLE);
        _userManager = (UserManager) smanager.lookup(UserManager.ROLE);
    }
    
    /**
     * Determines the skin directory is locked. If no, the lock owner is set in JSON map request attribute
     * @param request The request 
     * @param skinDir The skin directory
     * @return &lt;code&gt;true&lt;/code&gt; if the current user can write.
     * @throws IOException if an exception occurs while manipulating the skin directory 
     */
    protected boolean checkLock (Request request, Path skinDir) throws IOException
    {
        if (!_lockManager.canWrite(skinDir))
        {
            Map<String, Object> result = new HashMap<>();
            
            UserIdentity lockOwner = _lockManager.getLockOwner(skinDir);
            User user = _userManager.getUser(lockOwner.getPopulationId(), lockOwner.getLogin());

            result.put("isLocked", true);
            result.put("lockOwner", user != null ? user.getFullName() + " (" + lockOwner + ")" : lockOwner);
            
            request.setAttribute(JSonReader.OBJECT_TO_READ, result);
            return false;
        }
        return true;
    }
    
    /**
     * Determines if the model still exists
     * @param request The request
     * @param modelName The model name
     * @return <code>true</code> if the model still exists.
     */
    protected boolean checkModelExists (Request request, String modelName)
    {
        Map<String, Object> result = new HashMap<>();
        
        boolean modelExists = _modelsManager.getModel(modelName) != null;
        if (!modelExists)
        {
            result.put("unknownModel", true);
            result.put("modelName", modelName);
            
            request.setAttribute(JSonReader.OBJECT_TO_READ, result);
            return false;
        }
        
        return true;
    }
}
