/*
 *  Copyright 2011 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.skinfactory.generators;

import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.components.source.impl.SitemapSource;
import org.apache.cocoon.generation.ServiceableGenerator;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.collections.SetUtils;
import org.apache.excalibur.source.Source;
import org.apache.excalibur.source.SourceResolver;
import org.apache.excalibur.xml.dom.DOMParser;
import org.apache.excalibur.xml.sax.SAXParser;
import org.apache.excalibur.xml.xpath.XPathProcessor;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

import org.ametys.core.util.IgnoreRootHandler;
import org.ametys.plugins.skincommons.SkinEditionHelper;
import org.ametys.skinfactory.SkinFactoryComponent;
import org.ametys.skinfactory.parameters.AbstractSkinParameter;
import org.ametys.web.skin.SkinModel;
import org.ametys.web.skin.SkinModelsManager;

/**
 * SAX ribbon definition for skin parameters
 *
 */
public class RibbonGenerator extends ServiceableGenerator
{
    private SourceResolver _srcResolver;
    private SkinModelsManager _modelsManager;
    private SkinFactoryComponent _skinFactoryManager;
    private DOMParser _domParser;
    private XPathProcessor _xpathProcessor;
    private SkinEditionHelper _skinHelper;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _srcResolver = (SourceResolver) smanager.lookup(SourceResolver.ROLE);
        _skinHelper = (SkinEditionHelper) smanager.lookup(SkinEditionHelper.ROLE);
        _modelsManager = (SkinModelsManager) smanager.lookup(SkinModelsManager.ROLE);
        _skinFactoryManager = (SkinFactoryComponent) smanager.lookup(SkinFactoryComponent.ROLE);
        _domParser = (DOMParser) smanager.lookup(DOMParser.ROLE);
        _xpathProcessor = (XPathProcessor) smanager.lookup(XPathProcessor.ROLE);
    }
    
    @Override
    public void generate() throws IOException, SAXException, ProcessingException
    {
        String skinName = parameters.getParameter("skinName", null);
        
        String modelName = _skinHelper.getTempModel(skinName);
        if (modelName == null)
        {
            throw new IllegalArgumentException("The current skin has no model");
        }
        
        SkinModel model = _modelsManager.getModel(modelName);
        if (model == null)
        {
            throw new IllegalArgumentException("The model '" + modelName + "' does not exit anymore");
        }
        
        contentHandler.startDocument();
        XMLUtils.startElement(contentHandler, "cms");
        
        Source src = null;
        try
        {
            Path ribbonFile = model.getPath().resolve("model/cms-ribbon.xml");
            if (Files.exists(ribbonFile))
            {
                if (_isUpToDate(ribbonFile, modelName))
                {
                    SAXParser saxParser = null;
                    try (InputStream is = Files.newInputStream(ribbonFile))
                    {
                        saxParser = (SAXParser) manager.lookup(SAXParser.ROLE);
                        saxParser.parse(new InputSource(is), new IgnoreRootHandler(contentHandler));
                    }
                    catch (ServiceException e)
                    {
                        throw new ProcessingException("Unable to get a SAX parser", e);
                    }
                    finally
                    {
                        manager.release(saxParser);
                    }
                }
                else
                {
                    src = _srcResolver.resolveURI("cocoon:/" + modelName + "/update/cms-ribbon.xml");
                    ((SitemapSource) src).toSAX(new IgnoreRootHandler(contentHandler));
                    
                    // Update file
                    try (InputStream is = Files.newInputStream(ribbonFile))
                    {
                        Files.copy(is, ribbonFile, StandardCopyOption.REPLACE_EXISTING);
                    }
                }
            }
            else
            {
                src = _srcResolver.resolveURI("cocoon:/" + modelName + "/create/cms-ribbon.xml");
                ((SitemapSource) src).toSAX(new IgnoreRootHandler(contentHandler));
                
                // Create file
                try (InputStream is = Files.newInputStream(ribbonFile))
                {
                    Files.copy(is, ribbonFile);
                }
            }
        }
        finally
        {
            if (src != null)
            {
                _srcResolver.release(src);
            }
        }
        
        XMLUtils.startElement(contentHandler, "parameters");
        Map<String, AbstractSkinParameter> modelParameters = _skinFactoryManager.getModelParameters(modelName);
        for (AbstractSkinParameter parameter : modelParameters.values())
        {
            parameter.toSAX(contentHandler, modelName);
        }
        XMLUtils.endElement(contentHandler, "parameters");

        XMLUtils.endElement(contentHandler, "cms");
        contentHandler.endDocument();

    }

    private boolean _isUpToDate (Path ribbonFile, String modelName) throws IOException, SAXException
    {
        Set<String> ribbonParams = new HashSet<>();
        
        try (InputStream is = Files.newInputStream(ribbonFile))
        {
            Document document = _domParser.parseDocument(new InputSource(is));
            NodeList paramNodes = _xpathProcessor.selectNodeList(document, "/ribbon/tab/groups/group//parameter");
            
            for (int i = 0; i < paramNodes.getLength(); i++)
            {
                Element consistency = (Element) paramNodes.item(i);
                String id = consistency.getAttribute("id");
                
                ribbonParams.add(id);
            }
            
            return SetUtils.isEqualSet(ribbonParams, _skinFactoryManager.getModelParameters(modelName).keySet());
        }
    }
}
