/*
 *  Copyright 2011 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.skinfactory.parameters;

import java.io.IOException;
import java.io.OutputStream;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.HashMap;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.io.IOUtils;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.web.skin.SkinModel;

/**
 * Implementation of {@link AbstractSkinParameter} for a i18niable text
 */
public class TextParameter extends AbstractSkinParameter
{
    private String _defaultValue;
    private Path _xslFile;
    
    /**
     * Constructor
     * @param id the unique id
     * @param label the label
     * @param description the description
     * @param xslFile the XSL file
     * @param defaultValue the default value in the default language
     */
    public TextParameter(String id, I18nizableText label, I18nizableText description, Path xslFile, String defaultValue)
    {
        super(id, label, description);
        _xslFile = xslFile;
        _defaultValue = defaultValue;
    }
    
    /**
     * Get the CSS file path
     * @return the CSS file path
     */
    public Path getPath ()
    {
        return _xslFile;
    }
    
    @Override
    public SkinParameterType getType()
    {
        return SkinParameterType.TEXT;
    }
    
    @Override
    public void apply(Path tempDir, Path modelDir, Object value, String lang)
    {
        String relPath = modelDir.relativize(_xslFile).toString();
        Path cssFile = tempDir.resolve(relPath);
        String string = _filetoString(cssFile);
        
        Pattern pattern = Pattern.compile("^[^>]*>([^<]*)<!--\\s*AMETYS\\s*\\(\\s*\"([^\"]+)\"\\s*(?:,\\s*([^,\"\\s]+|\"[^\"]*\")\\s*)?(?:,\\s*([^,\"\\s]+|\"[^\"]*\")\\s*)?\\)\\s*-->.*$$", Pattern.MULTILINE);
        Matcher matcher = pattern.matcher(string);
        
        try (OutputStream os = Files.newOutputStream(cssFile))
        {
            int startIndex = 0;
            
            while (matcher.find())
            {
                int propertyStart = matcher.start(1);
                
                String beforeValue = string.substring(startIndex, propertyStart);
                IOUtils.write(beforeValue, os, "UTF-8");
                IOUtils.write((String) value, os, "UTF-8");
                
                startIndex = matcher.end(1);
            }
            
            IOUtils.write(string.substring(startIndex), os, "UTF-8");
        }
        catch (IOException e)
        {
            throw new SkinParameterException ("Unable to apply text parameter '" + getId() + "'", e);
        }
    }
    
    private String _filetoString (Path file)
    {
        try
        {
            return Files.readString(file, StandardCharsets.UTF_8);
        }
        catch (IOException e)
        {
            throw new SkinParameterException ("Unable to parse file '" + file.getFileName().toString() + "'", e);
        }
    }
    
    @Override
    public void toSAX(ContentHandler contentHandler, String modelName) throws SAXException
    {
        AttributesImpl attrs = new AttributesImpl();
        attrs.addCDATAAttribute("id", getId());
        attrs.addCDATAAttribute("type", SkinParameterType.TEXT.name().toLowerCase());
        
        XMLUtils.startElement(contentHandler, "parameter", attrs);
        
        getLabel().toSAX(contentHandler, "label");
        getDescription().toSAX(contentHandler, "description");
        
        XMLUtils.endElement(contentHandler, "parameter");
        
    }
    
    @Override
    public Map<String, Object> toJson(String modelName)
    {
        Map<String, Object> jsonObject = new HashMap<>();
        
        jsonObject.put("id", getId());
        jsonObject.put("type", SkinParameterType.TEXT.name().toLowerCase());
        jsonObject.put("label", getLabel());
        jsonObject.put("description", getDescription());
        
        return jsonObject;
    }
    
    @Override
    public String getDefaultValue(SkinModel model)
    {
        return _defaultValue;
    }
    
    @Override
    public String getDefaultValue(SkinModel model, String lang)
    {
        return getDefaultValue(model);
    }
}
