/*
 *  Copyright 2014 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.socialnetworking;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.acting.ServiceableAction;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.SourceResolver;
import org.apache.commons.lang3.StringUtils;

import org.ametys.core.captcha.CaptchaHelper;
import org.ametys.core.cocoon.ActionResultGenerator;
import org.ametys.core.ui.mail.StandardMailBodyHelper;
import org.ametys.core.ui.mail.StandardMailBodyHelper.MailBodyBuilder;
import org.ametys.core.util.I18nUtils;
import org.ametys.core.util.mail.SendMailHelper;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.web.cache.PageHelper;
import org.ametys.web.repository.page.Page;

import jakarta.mail.MessagingException;

/**
 * Check if the form is well formed
 * Send the mail to all the receiver
 */
public class SendMailAction extends ServiceableAction
{
    /** The pattern to check emails */
    protected static final Pattern EMAIL_VALIDATOR = Pattern.compile("^([a-z0-9._-]+@[a-z0-9.-]{2,}[.][a-zA-Z0-9]{2,})?$");
    
    /** The pattern to check a list of emails */
    protected static final Pattern EMAIL_LIST_VALIDATOR = Pattern.compile("^([a-z0-9._-]+@[a-z0-9.-]{2,}[.][a-z]{2,})?(\\s*,\\s*([a-z0-9._-]+@[a-z0-9.-]{2,}[.][a-z]{2,})?)*$");
    
    /** The pattern to check text input */
    protected static final Pattern TEXT_VALIDATOR = Pattern.compile("^\\s*$");
    
    /** Repository content */
    protected AmetysObjectResolver _resolver;
    
    /** The i18n utils of runtime */
    protected I18nUtils _i18nUtils;

    /** The page helper */
    protected PageHelper _pageHelper;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
        _i18nUtils = (I18nUtils) smanager.lookup(I18nUtils.ROLE);
        _pageHelper = (PageHelper) smanager.lookup(PageHelper.ROLE);
    }
    
    @SuppressWarnings("unchecked")
    @Override
    public Map act(Redirector redirector, SourceResolver resolver, Map objectModel, String source, Parameters parameters) throws Exception
    {
        Map<String, Object> result = new HashMap<>();
        List<I18nizableText> errors = new ArrayList<>();
        Request request = ObjectModelHelper.getRequest(objectModel);
        
        String pageId = _getPageId(request);
        Page page = _resolver.resolveById(pageId);
        
        if (_pageHelper.isCaptchaRequired(page))
        { 
            String captchaKey = request.getParameter("captcha-key");
            String captchaValue = request.getParameter("captcha-value");
            if (!CaptchaHelper.checkAndInvalidate(captchaKey, captchaValue)) 
            {
                errors.add(new I18nizableText("plugin.socialnetworking", "PLUGINS_SOCIAL_NETWORKING_SEND_MAIL_ERROR_CAPTCHA"));
            }
        }
        
        String emailBy = request.getParameter("email");
        if (emailBy == null || !EMAIL_VALIDATOR.matcher(StringUtils.trimToEmpty(emailBy)).matches() || TEXT_VALIDATOR.matcher(StringUtils.trimToEmpty(emailBy)).matches())
        {
            errors.add(new I18nizableText("plugin.socialnetworking", "PLUGINS_SOCIAL_NETWORKING_SEND_MAIL_ERROR_MAILBY"));
        }
        else
        {
            result.put("emailBy", emailBy);
        }
        
        String emailTo = request.getParameter("emailTo");
        if (emailTo == null || !EMAIL_LIST_VALIDATOR.matcher(StringUtils.trimToEmpty(emailTo)).matches() || TEXT_VALIDATOR.matcher(StringUtils.trimToEmpty(emailTo)).matches())
        {
            errors.add(new I18nizableText("plugin.socialnetworking", "PLUGINS_SOCIAL_NETWORKING_SEND_MAIL_ERROR_MAILTO"));
        }
        else
        {
            result.put("emailTo", emailTo);
        }
        
        String userName = request.getParameter("name");
        if (userName == null || TEXT_VALIDATOR.matcher(StringUtils.trimToEmpty(userName)).matches())
        {
            errors.add(new I18nizableText("plugin.socialnetworking", "PLUGINS_SOCIAL_NETWORKING_SEND_MAIL_ERROR_NAME"));
        }
        else
        {
            result.put("name", userName);
        }
        
        String text = request.getParameter("text");
        if (text == null)
        {
            errors.add(new I18nizableText("plugin.socialnetworking", "PLUGINS_SOCIAL_NETWORKING_SEND_MAIL_ERROR_CONTENT"));
        }
        else
        {
            result.put("text", text);
        }
        
        if (errors.isEmpty())
        {            
            String pagePath = request.getHeader("Referer");
            String lang = page.getSitemapName();
            
            List<String> i18nparam = new ArrayList<>();
            i18nparam.add(page.getTitle());
            i18nparam.add(page.getSite().getTitle());
            
            I18nizableText i18nTextSubject = new I18nizableText("plugin.socialnetworking", "PLUGINS_SOCIAL_NETWORKING_SEND_MAIL_SUBJECT", i18nparam);
            String subject = _i18nUtils.translate(i18nTextSubject, lang);
            
            i18nparam = new ArrayList<>();
            i18nparam.add(userName);
            i18nparam.add(pagePath); // Page path
            i18nparam.add(text);
            i18nparam.add(page.getSite().getUrl()); // Site url
            
            I18nizableText i18nTextBody = new I18nizableText("plugin.socialnetworking", "PLUGINS_SOCIAL_NETWORKING_SEND_MAIL_TEXT_BODY", i18nparam);
            String textBody = _i18nUtils.translate(i18nTextBody, lang);
            
            MailBodyBuilder bodyBuilder = StandardMailBodyHelper.newHTMLBody()
                .withMessage(new I18nizableText("plugin.socialnetworking", "PLUGINS_SOCIAL_NETWORKING_SEND_MAIL_HTML_BODY", i18nparam))
                .withLanguage(lang)
                .withTitle(subject)
                
                .withLink(page.getSite().getUrl(), new I18nizableText("plugin.socialnetworking", "PLUGINS_SOCIAL_NETWORKING_SEND_MAIL_HTML_BODY_SITE_LINK"));
            
            if (StringUtils.isNotEmpty(text))
            {
                bodyBuilder.withDetails(new I18nizableText("plugin.socialnetworking", "PLUGINS_SOCIAL_NETWORKING_SEND_MAIL_HTML_BODY_MESSAGE"), text, false);
            }
                        
            String htmlBody = bodyBuilder.build();
            
            String[] tabMail = StringUtils.split(emailTo, ",");
            for (int i = 0; i < tabMail.length; i++)
            {
                try
                {
                    SendMailHelper.newMail()
                                  .withSubject(subject)
                                  .withHTMLBody(htmlBody)
                                  .withTextBody(textBody)
                                  .withSender(emailBy)
                                  .withRecipient(StringUtils.trimToEmpty(tabMail[i]))
                                  .sendMail();
                }
                catch (MessagingException | IOException e)
                {
                    if (!result.containsKey("mailError"))
                    {
                        result.put("mailError", new ArrayList<>());
                    }
                    
                    ((List<String>) result.get("mailError")).add(tabMail[i]);
                    
                    getLogger().error("Failed to send mail to '" + tabMail[i] + "'", e);
                }
            }     
        }
        
        result.put("error", errors);
        
        request.setAttribute(ActionResultGenerator.MAP_REQUEST_ATTR, result);
        
        return EMPTY_MAP;
    }

    private String _getPageId(Request request)
    {
        String pageId = request.getParameter("page-id");
        if (pageId == null) 
        {
            throw new IllegalStateException("Unable to send page by mail: cannot determine the current page");
        }
        else
        {
            Page page = _resolver.resolveById(pageId);
            if (page == null)
            {
                throw new IllegalStateException("Unable to send page by mail: the page of id " + pageId + " is unknown");
            }
        }
        return pageId;
    }

}
