/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * The survey DAO.
 */
 Ext.define('Ametys.cms.survey.SurveyDAO', {
 	singleton: true,
 	
 	constructor: function(config)
 	{
		/**
		 * @callable
		 * @member Ametys.cms.survey.SurveyDAO
		 * @method createSurvey 
		 * Creates a survey.
		 * This calls the method 'createSurvey' of the server DAO 'SurveyDAO'.
		 * @param {Object[]} parameters The parameters to transmit to the server method
		 * @param {String} parameters.label The label
 		 * @param {String} parameters.title The title
 		 * @param {String} parameters.description The description
 		 * @param {String} parameters.endingMessage The ending message
 		 * @param {String} parameters.private A string representing a boolean indicating if the survey is private
 		 * @param {String} parameters.siteName The site name
 		 * @param {String} parameters.language The language
		 * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
		 * @param {Object} callback.returnedValue The value return from the server. Null on error (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
		 * @param {Object} callback.arguments Other arguments specified in option.arguments                 
		 * @param {Object} [options] Advanced options for the call.
		 * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
		 * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
		 * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
		 * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
		 * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
		 * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
		 * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
		 */
		this.addCallables(
			{
			    role: "org.ametys.plugins.survey.dao.SurveyDAO",
				methodName: "createSurvey",
	     		callback: {
	         		handler: this._createSurveyCb
	     		},
				errorMessage: {
				    msg: "{{i18n PLUGINS_SURVEY_CREATE_SURVEY_ERROR}}",
				    category: Ext.getClassName(this)
				}
			}
		);
		
		/**
		 * @callable
		 * @member Ametys.cms.survey.SurveyDAO
		 * @method editSurvey 
		 * Edits a survey.
		 * This calls the method 'editSurvey' of the server DAO 'SurveyDAO'.
		 * @param {Object[]} parameters The parameters to transmit to the server method
		 * @param {String} parameters.id The id of the survey to edit
		 * @param {String} parameters.label The label
 		 * @param {String} parameters.title The title
 		 * @param {String} parameters.description The description
 		 * @param {String} parameters.endingMessage The ending message
 		 * @param {String} parameters.private A string representing a boolean indicating if the survey is private
 		 * @param {String} parameters.siteName The site name
 		 * @param {String} parameters.language The language
		 * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
		 * @param {Object} callback.returnedValue The value return from the server. Null on error (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
		 * @param {Object} callback.arguments Other arguments specified in option.arguments                 
		 * @param {Object} [options] Advanced options for the call.
		 * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
		 * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
		 * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
		 * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
		 * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
		 * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
		 * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
		 */
		this.addCallables(
			{
				role: "org.ametys.plugins.survey.dao.SurveyDAO",
	     		methodName: "editSurvey",
	     		callback: {
	         		handler: this._editSurveyCb
	     		},
	     		errorMessage: {
	     			msg: "{{i18n PLUGINS_SURVEY_EDIT_SURVEY_ERROR}}",
	     			category: Ext.getClassName(this)
	     		}
			}
		);
		
		/**
		 * @callable
		 * @member Ametys.cms.survey.SurveyDAO
		 * @method copySurvey 
		 * Copies and pastes a survey.
		 * This calls the method 'copySurvey' of the server DAO 'SurveyDAO'.
		 * @param {Object[]} parameters The parameters to transmit to the server method
		 * @param {String} parameters.surveyId The id of the survey to copy
		 * @param {String} parameters.label The label
		 * @param {String} parameters.title The title
		 * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
		 * @param {Object} callback.returnedValue The value return from the server. Null on error (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
		 * @param {Object} callback.arguments Other arguments specified in option.arguments                 
		 * @param {Object} [options] Advanced options for the call.
		 * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
		 * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
		 * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
		 * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
		 * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
		 * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
		 * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
		 */
		this.addCallables(
			{
				role: "org.ametys.plugins.survey.dao.SurveyDAO",
	     		methodName: "copySurvey",
	     		callback: {
	         		handler: this._copySurveyCb
	     		},
	     		errorMessage: {
	     			msg: "{{i18n PLUGINS_SURVEY_COPY_SURVEY_ERROR}}",
	     			category: Ext.getClassName(this)
	     		}
			}
		);
		
		/**
		 * @callable
		 * @member Ametys.cms.survey.SurveyDAO
		 * @method deleteSurvey 
		 * Deletes a survey.
		 * This calls the method 'deleteSurvey' of the server DAO 'SurveyDAO'.
		 * @param {Object[]} parameters The parameters to transmit to the server method
		 * @param {String} parameters.id The id of the survey to delete
		 * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
		 * @param {Object} callback.returnedValue The value return from the server. Null on error (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
		 * @param {Object} callback.arguments Other arguments specified in option.arguments                 
		 * @param {Object} [options] Advanced options for the call.
		 * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
		 * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
		 * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
		 * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
		 * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
		 * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
		 * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
		 */
		this.addCallables(
			{
			    role: "org.ametys.plugins.survey.dao.SurveyDAO",
	     		methodName: "deleteSurvey",
	     		callback: {
	         		handler: this._deleteSurveyCb
	     		},
	     		errorMessage: {
	     			msg: "{{i18n PLUGINS_SURVEY_DELETE_SURVEY_ERROR}}",
	     			category: Ext.getClassName(this)
	     		}
  			}
  		);
  		
  		/**
		 * @callable
		 * @member Ametys.cms.survey.SurveyDAO
		 * @method getSurvey 
		 * Gets properties of a survey.
		 * This calls the method 'getSurvey' of the server DAO 'SurveyDAO'.
		 * @param {Object[]} parameters The parameters to transmit to the server method
		 * @param {String} parameters.id The id of the survey to retrieve
		 * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
		 * @param {Object} callback.returnedValue The value return from the server. Null on error (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
		 * @param {Object} callback.arguments Other arguments specified in option.arguments                 
		 * @param {Object} [options] Advanced options for the call.
		 * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
		 * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
		 * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
		 * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
		 * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
		 * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
		 * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
		 */
		this.addCallables(
			{
			    role: "org.ametys.plugins.survey.dao.SurveyDAO",
	     		methodName: "getSurvey",
	     		callback: {
	         		handler: Ext.emptyFn
	     		},
	     		errorMessage: {
	     			msg: "{{i18n PLUGINS_SURVEY_TOOL_ERROR}}",
	     			category: Ext.getClassName(this)
	     		}
  			}
  		);
  		
  		/**
		 * @callable
		 * @member Ametys.cms.survey.SurveyDAO
		 * @method isOnline 
		 * Gets the online status of a survey.
		 * This calls the method 'isOnline' of the server DAO 'SurveyDAO'.
		 * @param {Object[]} parameters The parameters to transmit to the server method
		 * @param {String} parameters.id The id of the survey
		 * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
		 * @param {Object} callback.returnedValue The value return from the server. Null on error (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
		 * @param {Object} callback.arguments Other arguments specified in option.arguments                 
		 * @param {Object} [options] Advanced options for the call.
		 * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
		 * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
		 * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
		 * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
		 * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
		 * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
		 * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
		 */
		this.addCallables(
			{
			    role: "org.ametys.plugins.survey.dao.SurveyDAO",
	     		methodName: "isOnline",
	     		callback: {
	         		handler: Ext.emptyFn
	     		},
	     		errorMessage: {
	     			msg: "{{i18n PLUGINS_SURVEY_REINIT_SURVEY_ONLINE_STATUS_ERROR}}",
	     			category: Ext.getClassName(this)
	     		}
  			}
  		);
  		
  		/**
		 * @callable
		 * @member Ametys.cms.survey.SurveyDAO
		 * @method getChildren 
		 * Gets the children pages of a survey.
		 * This calls the method 'getChildren' of the server DAO 'SurveyDAO'.
		 * @param {Object[]} parameters The parameters to transmit to the server method
		 * @param {String} parameters.id The id of the survey
		 * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
		 * @param {Object} callback.returnedValue The value return from the server. Null on error (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
		 * @param {Object} callback.arguments Other arguments specified in option.arguments                 
		 * @param {Object} [options] Advanced options for the call.
		 * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
		 * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
		 * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
		 * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
		 * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
		 * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
		 * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
		 */
		this.addCallables(
			{
			    role: "org.ametys.plugins.survey.dao.SurveyDAO",
	     		methodName: "getChildren",
	     		callback: {
	         		handler: Ext.emptyFn
	     		},
	     		errorMessage: {
	     			msg: "{{i18n PLUGINS_SURVEY_TOOL_ERROR}}",
	     			category: Ext.getClassName(this)
	     		}
  			}
  		);
  		
  		/**
		 * @callable
		 * @member Ametys.cms.survey.SurveyDAO
		 * @method validateSurvey 
		 * Validates a survey.
		 * This calls the method 'validateSurvey' of the server DAO 'SurveyDAO'.
		 * @param {Object[]} parameters The parameters to transmit to the server method
		 * @param {String} parameters.id The id of the survey to validate.
		 * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
		 * @param {Object} callback.returnedValue The value return from the server. Null on error (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
		 * @param {Object} callback.arguments Other arguments specified in option.arguments                 
		 * @param {Object} [options] Advanced options for the call.
		 * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
		 * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
		 * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
		 * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
		 * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
		 * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
		 * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
		 */
		this.addCallables(
			{
			    role: "org.ametys.plugins.survey.dao.SurveyDAO",
	     		methodName: "validateSurvey",
	     		callback: {
	         		handler: this._validateSurveyCb
	     		},
	     		errorMessage: {
	     			msg: "{{i18n PLUGINS_SURVEY_VALIDATE_SURVEY_ERROR}}",
	     			category: Ext.getClassName(this)
	     		}
  			}
  		);
  		
  		/**
		 * @callable
		 * @member Ametys.cms.survey.SurveyDAO
		 * @method reinitSurvey 
		 * Reinitializes a survey.
		 * This calls the method 'reinitSurvey' of the server DAO 'SurveyDAO'.
		 * @param {Object[]} parameters The parameters to transmit to the server method
		 * @param {String} parameters.id The id of the survey to validate.
		 * @param {Boolean} parameters.invalidate True to invalidate the survey
		 * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
         * @param {Object} callback.returnedValue The value return from the server. Null on error (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
         * @param {String} callback.returnedValue.id The survey id.
         * @param {String[]} [callback.returnedValue.modifiedPages] If parameters.invalidate is true, also returns the list of pages modified. 
		 * @param {Object} callback.arguments Other arguments specified in option.arguments                 
		 * @param {Object} [options] Advanced options for the call.
		 * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
		 * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
		 * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
		 * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
		 * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
		 * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
		 * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
		 */
		this.addCallables(
			{
			    role: "org.ametys.plugins.survey.dao.SurveyDAO",
	     		methodName: "reinitSurvey",
	     		callback: {
	         		handler: this._reinitSurveyCb
	     		},
	     		errorMessage: {
	     			msg: "{{i18n PLUGINS_SURVEY_REINIT_SURVEY_ERROR}}",
	     			category: Ext.getClassName(this)
	     		}
  			}
  		);
  		
  		/**
		 * @callable
		 * @member Ametys.cms.survey.SurveyDAO
		 * @method setRedirection 
		 * Sets a new redirection page to the survey.
		 * This calls the method 'setRedirection' of the server DAO 'SurveyDAO'.
		 * @param {Object[]} parameters The parameters to transmit to the server method
		 * @param {String} parameters.surveyId The id of the survey to edit.
		 * @param {String} parameters.pageId The id of the redirection page.
		 * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
		 * @param {Object} callback.returnedValue The value return from the server. Null on error (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
		 * @param {Object} callback.arguments Other arguments specified in option.arguments                 
		 * @param {Object} [options] Advanced options for the call.
		 * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
		 * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
		 * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
		 * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
		 * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
		 * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
		 * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
		 */
		this.addCallables(
			{
			    role: "org.ametys.plugins.survey.dao.SurveyDAO",
	     		methodName: "setRedirection",
	     		callback: {
	         		handler: this._setRedirectionCb
	     		},
	     		errorMessage: {
	     			msg: "{{i18n PLUGINS_SURVEY_SET_REDIRECTION_ERROR}}",
	     			category: Ext.getClassName(this)
	     		}
  			}
  		);
  		
  		/**
		 * @callable
		 * @member Ametys.cms.survey.SurveyDAO
		 * @method moveObject 
		 * Moves an element of the survey.
		 * This calls the method 'moveObject' of the server DAO 'SurveyDAO'.
		 * @param {Object[]} parameters The parameters to transmit to the server method
		 * @param {String} parameters.id The id of the element to move.
		 * @param {String} parameters.oldParent The id of the element's parent.
		 * @param {String} parameters.newParent The id of the new element's parent.
		 * @param {Number} parameters.index The index where to move. null to place the element at the end.
		 * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
		 * @param {Object} callback.returnedValue The value return from the server. Null on error (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
		 * @param {Object} callback.arguments Other arguments specified in option.arguments                 
		 * @param {Object} [options] Advanced options for the call.
		 * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
		 * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
		 * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
		 * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
		 * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
		 * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
		 * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
		 */
		this.addCallables(
			{
			    role: "org.ametys.plugins.survey.dao.SurveyDAO",
	     		methodName: "moveObject",
	     		callback: {
	         		handler: this._moveObjectCb
	     		},
	     		errorMessage: {
	     			msg: "{{i18n PLUGINS_SURVEY_TOOL_MOVE_ERROR}}",
	     			category: Ext.getClassName(this)
	     		}
  			}
  		);
  		
  		/**
		 * @callable
		 * @member Ametys.cms.survey.SurveyDAO
		 * @method addGrantedUsers 
		 * Stores a list of granted users and/or granted groups in a survey.
		 * This calls the method 'addGrantedUsers' of the server DAO 'SurveyDAO'.
		 * @param {Object[]} parameters The parameters to transmit to the server method
		 * @param {String} parameters.surveyId The id of the survey.
		 * @param {Object[]} parameters.grantedUsers The users to add.
		 * @param {String} parameters.grantedUsers.login The login of the user
		 * @param {String} parameters.grantedUsers.population The id of the population of the user
		 * @param {Object[]} parameters.grantedGroups The groups to add.
		 * @param {String} parameters.grantedGroups.id The id of the group
		 * @param {String} parameters.grantedGroups.groupDirectory The id of the group directory of the group
		 * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
		 * @param {Object} callback.returnedValue The value return from the server. Null on error (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
		 * @param {Object} callback.arguments Other arguments specified in option.arguments                 
		 * @param {Object} [options] Advanced options for the call.
		 * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
		 * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
		 * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
		 * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
		 * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
		 * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
		 * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
		 */
		this.addCallables(
			{
			    role: "org.ametys.plugins.survey.dao.SurveyDAO",
	     		methodName: "addGrantedUsers",
	     		callback: {
	         		handler: this._addGrantedUsersCb
	     		},
	     		errorMessage: {
	     			msg: "{{i18n PLUGINS_SURVEY_GRANTED_USERS_ADDUSERS_ERROR}}",
	     			category: Ext.getClassName(this)
	     		}
  			}
  		);
  		
  		/**
		 * @callable
		 * @member Ametys.cms.survey.SurveyDAO
		 * @method removeGrantedUsers 
		 * Removes a list of granted users and/or granted groups in a survey.
		 * This calls the method 'removeGrantedUsers' of the server DAO 'SurveyDAO'.
		 * @param {Object[]} parameters The parameters to transmit to the server method
		 * @param {String} parameters.surveyId The id of the survey.
         * @param {Object[]} parameters.grantedUsers The users to remove.
         * @param {String} parameters.grantedUsers.login The login of the user
         * @param {String} parameters.grantedUsers.population The id of the population of the user
         * @param {Object[]} parameters.grantedGroups The groups to remove.
         * @param {String} parameters.grantedGroups.id The id of the group
         * @param {String} parameters.grantedGroups.groupDirectory The id of the group directory of the group
		 * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
		 * @param {Object} callback.returnedValue The value return from the server. Null on error (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
		 * @param {Object} callback.arguments Other arguments specified in option.arguments                 
		 * @param {Object} [options] Advanced options for the call.
		 * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
		 * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
		 * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
		 * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
		 * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
		 * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
		 * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
		 */
		this.addCallables(
			{
			    role: "org.ametys.plugins.survey.dao.SurveyDAO",
	     		methodName: "removeGrantedUsers",
	     		callback: {
	         		handler: this._removeGrantedUsersCb
	     		},
	     		errorMessage: {
	     			msg: "{{i18n PLUGINS_SURVEY_GRANTED_USERS_REMOVE_ERROR}}",
	     			category: Ext.getClassName(this)
	     		}
  			}
  		);
  		
  		/**
		 * @callable
		 * @member Ametys.cms.survey.SurveyDAO
		 * @method sendInvitations 
		 * Sends invitations emails.
		 * This calls the method 'sendInvitations' of the server DAO 'SurveyDAO'.
		 * @param {Object[]} parameters The parameters to transmit to the server method
		 * @param {String} parameters.surveyId The id of the survey.
		 * @param {String} parameters.message The message content.
		 * @param {String} parameters.siteName The site name.
		 * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
		 * @param {Object} callback.returnedValue The value return from the server. Null on error (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
		 * @param {Object} callback.arguments Other arguments specified in option.arguments                 
		 * @param {Object} [options] Advanced options for the call.
		 * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
		 * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
		 * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
		 * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
		 * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
		 * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
		 * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
		 */
		this.addCallables(
			{
			    role: "org.ametys.plugins.survey.dao.SurveyDAO",
	     		methodName: "sendInvitations",
	     		callback: {
	         		handler: Ext.emptyFn
	     		},
	     		errorMessage: {
	     			msg: "{{i18n PLUGINS_SURVEY_SEND_MAIL_ERROR}}",
	     			category: Ext.getClassName(this)
	     		}
  			}
  		);
  		
  		/**
		 * @callable
		 * @member Ametys.cms.survey.SurveyDAO
		 * @method getStatistics 
		 * Generates statistics on each question of a survey.
		 * This calls the method 'getStatistics' of the server DAO 'SurveyDAO'.
		 * @param {Object[]} parameters The parameters to transmit to the server method
		 * @param {String} parameters.id The id of the survey.
		 * @param {Function} callback The function to call when the java process is over. Use options.scope for the scope. 
		 * @param {Object} callback.returnedValue The value return from the server. Null on error (please note that when an error occured, the callback may not be called depending on the value of errorMessage).
		 * @param {Object} callback.arguments Other arguments specified in option.arguments                 
		 * @param {Object} [options] Advanced options for the call.
		 * @param {Boolean/String/Object} [options.errorMessage] Display an error message. See Ametys.data.ServerComm#callMethod errorMessage.
		 * @param {Boolean/String/Object} [options.waitMessage] Display a waiting message. See Ametys.data.ServerComm#callMethod waitMessage.
		 * @param {Number} [options.scope] This parameter is the scope used to call the callback. Moreover is the given class is a mixin of Ametys.data.ServerCaller, its methods Ametys.data.ServerCaller#beforeServerCall and Ametys.data.ServerCaller#afterServerCall will be used so see their documentation to look for additional options (such a refreshing on Ametys.ribbon.element.ui.ButtonController#beforeServerCall).
		 * @param {Number} [options.priority] The message priority. See Ametys.data.ServerComm#callMethod for more information on the priority. PRIORITY_SYNCHRONOUS cannot be used here.
		 * @param {String} [options.cancelCode] Cancel similar unachieved read operations. See Ametys.data.ServerComm#callMethod cancelCode.
		 * @param {Object} [options.arguments] Additional arguments set in the callback.arguments parameter.                  
		 * @param {Boolean} [options.ignoreCallbackOnError] If the server throws an exception, should the callback beeing called with a null parameter. See Ametys.data.ServerComm#callMethod ignoreOnError.
		 */
		this.addCallables(
			{
			    role: "org.ametys.plugins.survey.dao.SurveyDAO",
	     		methodName: "getStatistics",
	     		callback: {
	         		handler: Ext.emptyFn
	     		},
	     		errorMessage: {
	     			msg: "{{i18n PLUGINS_SURVEY_STATISTICS_TOOL_QUERY_ERROR}}",
	     			category: Ext.getClassName(this)
	     		}
  			}
  		);
 	},
 	
 	/**
 	 * Callback function called after {@link #createSurvey} has been processed.
 	 * @param {Object} response The text response provided by the {@link Ametys.data.ServerComm}
 	 * @private
 	 */
 	_createSurveyCb: function(response)
 	{
    	Ext.create("Ametys.message.Message", {
    		type: Ametys.message.Message.CREATED,
    		targets: {
    		    id: Ametys.message.MessageTarget.SURVEY,
    			parameters: {
    				id: response.id
    			}
    		}
    	});
 	},
 	
 	/**
 	 * Callback function called after {@link #editSurvey} has been processed.
 	 * @param {Object} response The text response provided by the {@link Ametys.data.ServerComm}
 	 * @private
 	 */
 	_editSurveyCb: function(response)
 	{
		Ext.create("Ametys.message.Message", {
			type: Ametys.message.Message.MODIFIED,
			targets: {
			    id: Ametys.message.MessageTarget.SURVEY,
				parameters: {
					id: response.id
				}
			}
		});
 	},
 	
 	/**
 	 * Callback function called after {@link #copySurvey} has been processed.
 	 * @param {Object} response The text response provided by the {@link Ametys.data.ServerComm}
 	 * @private
 	 */
 	_copySurveyCb: function(response)
 	{
    	Ext.create("Ametys.message.Message", {
    		type: Ametys.message.Message.CREATED,
    		targets: {
    		    id: Ametys.message.MessageTarget.SURVEY,
    			parameters: {
    				id: response.id
    			}
    		}
    	});
 	},
 	
 	/**
 	 * Callback function called after {@link #deleteSurvey} has been processed.
 	 * @param {Object} response The text response provided by the {@link Ametys.data.ServerComm}
 	 * @private
 	 */
 	_deleteSurveyCb: function(response)
 	{
	    Ext.create("Ametys.message.Message", {
	    	type: Ametys.message.Message.DELETED,
	    	targets: {
	    	    id: Ametys.message.MessageTarget.SURVEY,
	    		parameters: {
					id: response.id
				}
	    	}
	    });
	 },
	 
	 /**
 	 * Callback function called after {@link #validateSurvey} has been processed.
 	 * @param {Object} response The text response provided by the {@link Ametys.data.ServerComm}
 	 * @private
 	 */
	 _validateSurveyCb: function(response)
 	{
		Ext.create("Ametys.message.Message", {
			type: Ametys.message.Message.WORKFLOW_CHANGED,
			targets: {
			    id: Ametys.message.MessageTarget.SURVEY,
				parameters: {
					id: response.id
				}
			}
		});
 	},
 	
 	/**
 	 * Callback function called after {@link #reinitSurvey} has been processed.
 	 * @param {Object} response The text response provided by the {@link Ametys.data.ServerComm}
 	 * @private
 	 */
 	_reinitSurveyCb: function(response)
 	{
		Ext.create("Ametys.message.Message", {
			type: Ametys.message.Message.WORKFLOW_CHANGED,
			targets: {
			    id: Ametys.message.MessageTarget.SURVEY,
				parameters: {
					id: response.id
				}
			}
		});
		
		if (response.modifiedPages)
	    {
		    Ext.create("Ametys.message.Message", {
	            type: Ametys.message.Message.MODIFIED,
	            targets: {
	                id: Ametys.message.MessageTarget.PAGE,
	                parameters: {
	                    ids: response.modifiedPages
	                }
	            }
	        });
	    }
 	},
	 
	 /**
 	 * Callback function called after {@link #addGrantedUsers} has been processed.
 	 * @param {Object} response The text response provided by the {@link Ametys.data.ServerComm}
 	 * @private
 	 */
	 _addGrantedUsersCb: function(response)
 	{
		Ext.create("Ametys.message.Message", {
			type: Ametys.message.Message.MODIFIED,
			targets: {
			    id: Ametys.message.MessageTarget.SURVEY,
				parameters: {
					id: response.surveyId
				}
			}
		});
 	},
 	
	 /**
 	 * Callback function called after {@link #removeGrantedUsers} has been processed.
 	 * @param {Object} response The text response provided by the {@link Ametys.data.ServerComm}
 	 * @private
 	 */
	 _removeGrantedUsersCb: function(response)
 	{
		Ext.create("Ametys.message.Message", {
			type: Ametys.message.Message.MODIFIED,
			targets: {
			    id: Ametys.message.MessageTarget.SURVEY,
				parameters: {
					id: response.surveyId
				}
			}
		});
 	},
 	
	 /**
 	 * Callback function called after {@link #setRedirection} has been processed.
 	 * @param {Object} response The text response provided by the {@link Ametys.data.ServerComm}
 	 * @private
 	 */
	 _setRedirectionCb: function(response)
 	{
		Ext.create("Ametys.message.Message", {
			type: Ametys.message.Message.MODIFIED,
			targets: {
			    id: Ametys.message.MessageTarget.SURVEY,
				parameters: {
					id: response.id
				}
			}
		});
 	},
 	
 	/**
 	 * Callback function called after {@link #moveObject} has been processed.
 	 * @param {Object} response The text response provided by the {@link Ametys.data.ServerComm}
 	 * @private
 	 */
 	_moveObjectCb: function(response)
 	{
 		var targetId, parameters;
 		if (response.type == 'page')
 		{
 		   targetId = Ametys.message.MessageTarget.SURVEY_PAGE;
 			parameters = {
 				id: response.id,
 				surveyId: response.newParentId
 			};
 		}
 		else if (response.type == 'question')
 		{
 		   targetId = Ametys.message.MessageTarget.SURVEY_QUESTION;
 			parameters = {
 				id: response.id,
 				type: response.questionType,
 				pageId: response.newParentId,
 				surveyId: response.newSurveyId
 			};
 		}
 		
		Ext.create("Ametys.message.Message", {
			type: Ametys.message.Message.MOVED,
			targets: {
			    id: targetId,
				parameters: parameters
			}
		});
 	}
 	
 });