/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
 
/**
 * Helper for creating or editing a FREE_TEXT question.
 */
Ext.define('Ametys.plugins.survey.question.InputTextDialog', {
	singleton: true,
 	
 	/**
 	 * @private
 	 * @property {String} _mode Can be 'new or 'edit'.
 	 */
 	
 	/**
 	 * @private
 	 * @property {Ametys.window.DialogBox} _box The dialog box for creating/editing a page.
 	 */
 	 
 	/**
 	 * @private
 	 * @property {Ext.form.Panel} _formPanel The form panel of the dialog box.
 	 */
 	 
 	 /**
 	  * @private
 	  * @property {Boolean} _initialized Indicates if the dialog box is initialized.
 	  */
 	
	/**
	 * Open the dialog to create or edit a question
	 * @param {Object} config The configuration object
	 * @param {Ametys.message.MessageTarget} config.target If 'new' mode, the parent target. If 'edit' mode, the target to edit.
	 * @param {String} [config.mode=new] The mode: 'new' for creation or 'edit' for edition
	 */
 	open: function (config)
 	{
 		this._mode = config.mode || 'new';
		
		this._delayedInitialize();
		
		this._initForm (config.target);
 	},
 	
 	/**
	 * @private
	 * Creates the dialog box
	 */
	_delayedInitialize: function ()
	{
		if (!this._initialized)
		{
			this._formPanel = this._createFormPanel();
			
			this._box = Ext.create('Ametys.window.DialogBox', {
				title: "{{i18n PLUGINS_SURVEY_DIALOG_INPUT_TEXT}}",
	            iconCls: "ametysicon-input",
	            
	            width: 550,
		        scrollable: false,
	            
	            items: [this._formPanel],
	            
	            closeAction: 'hide',
	            
	            referenceHolder: true,
	            defaultButton: 'validate',
	            defaultFocus: 'label', 
	            selectDefaultFocus: true,
	            
	            buttons: [{
	            	reference: 'validate',
	                text: "{{i18n PLUGINS_SURVEY_DIALOG_BUTTON_OK}}",
	                handler: Ext.bind(this._validate, this)
	            }, {
	                text: "{{i18n PLUGINS_SURVEY_DIALOG_BUTTON_CANCEL}}",
	                handler: Ext.bind(this._cancel, this)
	            }]    
			});
			
			this._initialized = true;
		}
	},
	
	/**
    * @private
    * Handler for the 'ok' button of the dialog box
    */
    _validate: function ()
    {
        var form = this._formPanel.getForm();
        if (!form.isValid())
        {
        	return;
        }
        
        var values = form.getValues();
        if (values.multiline)
        {
        	values.regexp = '';
        }
        
        values.type = values.multiline ? 'MULTILINE_FREE_TEXT' : 'FREE_TEXT';
 		
 		if (this._mode == 'new')
 		{
 			var params = [values];
 			Ametys.cms.survey.QuestionDAO.createQuestion(params, this._validateCb, {scope: this} );
 		}
 		else
 		{
 			var id = form.findField('id').getValue();
 			
 			var params = [values];
 			Ametys.cms.survey.QuestionDAO.editQuestion(params, this._validateCb, {scope: this} );
 		}
    },
    
    /**
     * Callback function called after creation or edition process
     * @param {Object} response The server response
     * @private
     */
    _validateCb: function (response)
    {
    	if (!response['error'])
    	{
    		this._box.close();
    	}
    },
    
    /**
     * @private
     * Callback for the "cancel" button of the dialog. Close the dialog.
     */
    _cancel: function ()
    {
        this._box.close();
    },
 	
 	/**
 	 * @private
 	 * Initializes the form with some optional values.
 	 * @param {Ametys.message.MessageTarget} target If 'new' mode, the parent target. If 'edit' mode, the target to edit.
 	 */
	_initForm: function (target)
 	{
 		var form = this._formPanel.getForm();
		form.reset();
 		
 		if (this._mode === "new") 
        {
        	form.findField('id').setValue('');
 			form.findField('pageId').setValue(target.getParameters().id);
 			
 			form.findField('label').setValue("{{i18n PLUGINS_SURVEY_DIALOG_QUESTION_LABEL_DEFAULT}}");
			form.findField('title').setValue('');
			form.findField('title').clearInvalid();
			form.findField('mandatory').setValue(false);
			form.findField('multiline').setValue(false);
			form.findField('regexp').setValue('');
			form.findField('regexp').enable();
			form.findField('picture-alternative').setValue();
			form.findField('picture').setValue();
			
			this._box.show();
        }
 		else
 		{
 			form.findField('id').setValue(target.getParameters().id);
 			form.findField('pageId').setValue('');
 			
 			Ametys.cms.survey.QuestionDAO.getQuestion([target.getParameters().id], this._setValues, {scope: this});
 		}
 	},
 	
 	/**
 	 * Fill the form with some values.
 	 * @param {Object} data The definition.
 	 * @private
 	 */
 	_setValues: function(data)
 	{
 		var form = this._formPanel.getForm();
 		
	    form.findField('label').setValue(data.label);
	    form.findField('title').setValue(data.title);
	    form.findField('mandatory').setValue(data.mandatory == 'true');
	    form.findField('multiline').setValue(data.type == 'MULTILINE_FREE_TEXT');
		form.findField('regexp').setValue(data.regexp);
		form.findField('regexp').setDisabled(data.type == 'MULTILINE_FREE_TEXT');
		
		form.findField('picture-alternative').setValue(data.pictureAlternative);
		    
	    if (!Ext.Object.isEmpty(data.picture))
    	{
		    data.picture.id = 'untouched';
		    form.findField('picture').setValue(data.picture);
    	}
	    else
	    {
	    	form.findField('picture').setValue();
	    }
		    
		this._box.show();
 	},
 	
 	/**
 	 * Creates the form panel of this dialog box.
 	 * @return {Ext.form.Panel} The form panel
 	 * @private
 	 */
 	_createFormPanel: function()
 	{
 		var regexpStore = Ext.create('Ext.data.Store', {
 			fields: ['value', 'displayText'],
 			data: [
 				['', "{{i18n PLUGINS_SURVEY_DIALOG_QUESTION_REGEXPTYPE_TEXT}}"],
 				['int', "{{i18n PLUGINS_SURVEY_DIALOG_QUESTION_REGEXPTYPE_INT}}"],
 				['float', "{{i18n PLUGINS_SURVEY_DIALOG_QUESTION_REGEXPTYPE_FLOAT}}"],
 				['email', "{{i18n PLUGINS_SURVEY_DIALOG_QUESTION_REGEXPTYPE_EMAIL}}"],
 				['phone', "{{i18n PLUGINS_SURVEY_DIALOG_QUESTION_REGEXPTYPE_PHONE}}"]
 			]
 		});
 		
 		var formPanel = Ext.create('Ext.form.Panel', {
	        defaultType: 'textfield',
	        defaults: {
	        	cls: 'ametys',
	        	labelSeparator: '',
	        	labelAlign: 'right',
	        	labelWidth: 110,
	        	width: '100%',
	        	msgTarget: 'side'
	        },
	        
	        border: false,
	        scrollable: true,
	        
	        items: [
		        {
		        	xtype: 'hidden',
		        	name: 'id'
		        },
		        {
		        	xtype: 'hidden',
		        	name: 'pageId'
		        },
		        {
		        	itemId: 'label',
		        	name: 'label',
		        	fieldLabel: "{{i18n PLUGINS_SURVEY_DIALOG_QUESTION_LABEL}}",
	                ametysDescription: "{{i18n PLUGINS_SURVEY_DIALOG_QUESTION_LABEL_DESC}}",
	                allowBlank: false,
	                maxLength: 20
		        },
		        {
		        	name: 'title',
	                fieldLabel: "{{i18n PLUGINS_SURVEY_DIALOG_QUESTION_TITLE}}",
	                ametysDescription: "{{i18n PLUGINS_SURVEY_DIALOG_QUESTION_TITLE_DESC}}",
	                allowBlank: false
		        },
		        {
                    xtype: 'edition.file',
                    name: 'picture',
                    allowSources: [Ametys.form.widget.File.External.SOURCE, Ametys.cms.widget.File.Resource.SOURCE],
                    filter: Ametys.form.widget.File.IMAGE_FILTER,
                    fieldLabel: "{{i18n PLUGINS_SURVEY_DIALOG_QUESTION_PICTURE_LABEL}}",
                    ametysDescription: "{{i18n PLUGINS_SURVEY_DIALOG_QUESTION_PICTURE_DESC}}"
                },
                {
                    xtype: 'textfield',
                    name: 'picture-alternative',
                    fieldLabel: "{{i18n PLUGINS_SURVEY_DIALOG_QUESTION_PICTURE_ALT_LABEL}}",
                    ametysDescription: "{{i18n PLUGINS_SURVEY_DIALOG_QUESTION_PICTURE_ALT_DESC}}"
                },
		        {
		 			xtype: 'combobox',
			        store: regexpStore,
			        name: 'regexp',
					fieldLabel : "{{i18n PLUGINS_SURVEY_DIALOG_QUESTION_REGEXP}}",
					ametysDescription: "{{i18n PLUGINS_SURVEY_DIALOG_QUESTION_REGEXP_DESC}}",
					
					emptyText: "{{i18n PLUGINS_SURVEY_DIALOG_QUESTION_REGEXPTYPE_TEXT}}",
					queryMode: 'local',
			        allowBlank: true,
			        forceSelection: true,
			        triggerAction: 'all',
			        
			        valueField: 'value',
		        	displayField: 'displayText'
		 		},
		        {
		        	xtype: 'checkbox',
		        	name: 'mandatory',
		        	fieldLabel: ' ',
	                boxLabel: "{{i18n PLUGINS_SURVEY_DIALOG_QUESTION_MANDATORY}}",
	                ametysDescription: "{{i18n PLUGINS_SURVEY_DIALOG_QUESTION_MANDATORY_DESC}}",
	                inputValue: 'true',
	                checked: false
		        },
		        {
		        	xtype: 'checkbox',
		        	name: 'multiline',
		        	fieldLabel: ' ',
	                boxLabel: "{{i18n PLUGINS_SURVEY_DIALOG_QUESTION_MULTILINE}}",
	                ametysDescription: "{{i18n PLUGINS_SURVEY_DIALOG_QUESTION_MULTILINE_DESC}}",
	                checked: false,
	                listeners: [{'change': Ext.bind(this._onSelectMultiline, this)}]
		        }
	        ]
        	
 		});
 		
 		return formPanel;
 	},
 	
 	/**
 	 * Disables/enables the 'regexp' field depending on the 'multiline' field check status.
 	 * @param {Ext.form.field.Field} field The field
     * @param {Object} newValue The new value
     * @param {Object} oldValue The original value
 	 * @private
 	 */
 	_onSelectMultiline: function(field, newValue, oldValue)
 	{
 		var form = this._formPanel.getForm();
 		if (newValue)
		{
			form.findField('regexp').setValue('');
			form.findField('regexp').disable();
		}
		else
		{
			form.findField('regexp').enable();
		}
 	}
 	
 	
});