/*
 *  Copyright 2013 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.survey;

import org.apache.avalon.framework.logger.LogEnabled;
import org.apache.avalon.framework.logger.Logger;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.avalon.framework.thread.ThreadSafe;

import org.ametys.cms.repository.Content;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.ModifiableTraversableAmetysObject;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.survey.repository.Survey;
import org.ametys.plugins.survey.repository.SurveyPage;
import org.ametys.plugins.survey.repository.SurveyQuestion;
import org.ametys.plugins.survey.repository.SurveyRule;
import org.ametys.plugins.survey.repository.SurveyRule.RuleType;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.sitemap.Sitemap;
import org.ametys.web.site.CopyUpdater;

/**
 * Updates copied surveys, updating the rules.
 */
public class SurveyCopyUpdater implements CopyUpdater, ThreadSafe, Serviceable, LogEnabled
{
    private AmetysObjectResolver _resolver;
    private Logger _logger;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
    }
    
    @Override
    public void enableLogging(Logger logger)
    {
        _logger = logger;
    }

    @Override
    public void updateSite(Site initialSite, Site createdSite)
    {
        ModifiableTraversableAmetysObject pluginsNode = createdSite.getRootPlugins();
        
        for (Sitemap sitemap : createdSite.getSitemaps())
        {
            try
            {
                ModifiableTraversableAmetysObject surveysNode = pluginsNode.getChild("survey/ametys:surveys/" + sitemap.getSitemapName());
                
                AmetysObjectIterable<Survey> surveys = surveysNode.getChildren();
                for (Survey survey : surveys)
                {
                    for (SurveyPage cPage : survey.getPages())
                    {
                        _updateRulesAfterCopy(survey, cPage);
                    }
                }
            }
            catch (UnknownAmetysObjectException e)
            {
                // Nothing to do
            }
            
        }
    }
    
    private void _updateRulesAfterCopy (Survey cSurvey, SurveyPage cPage)
    {
        if (cPage.hasRule())
        {
            SurveyRule rule = cPage.getRule();
            String pageId = rule.getPage();
            if (pageId != null)
            {
                try
                {
                    // Find symmetric page in created survey
                    AmetysObject originalPage = _resolver.resolveById(pageId);
                    AmetysObject symmetricPage = cSurvey.getChild(originalPage.getName());
                    cPage.setRule(rule.getType(), symmetricPage.getId());
                }
                catch (UnknownAmetysObjectException e)
                {
                    // Delete the rule
                    _logger.warn("Symmetric page has not found during copy. The rule is deleted.");
                    cPage.deleteRule();
                }
            }
        }
        
        for (SurveyQuestion cQuestion : cPage.getQuestions())
        {
            _updateRulesAfterCopy (cSurvey, cQuestion);
        }
    }
    
    private void _updateRulesAfterCopy (Survey cSurvey, SurveyQuestion cQuestion)
    {
        for (SurveyRule cRule : cQuestion.getRules())
        {
            String pageId = cRule.getPage();
            if (pageId != null)
            {
                String option = cRule.getOption();
                try
                {
                    // Find symmetric page in created survey
                    AmetysObject originalPage = _resolver.resolveById(pageId);
                    AmetysObject symmetricPage = cSurvey.getChild(originalPage.getName());
                    
                    RuleType type = cRule.getType();
                    cQuestion.deleteRule(option);
                    cQuestion.addRules(option, type, symmetricPage.getId());
                }
                catch (UnknownAmetysObjectException e)
                {
                    // Delete rule
                    _logger.warn("Symmetric page has not found during copy. The rule is deleted.");
                    cQuestion.deleteRule(option);
                }
            }
        }
    }
    
    @Override
    public void updateContent(Site initialSite, Site createdSite, Content initialContent, Content createdContent)
    {
        // Nothing to do
    }

    @Override
    public void updatePage(Site initialSite, Site createdSite, Page page)
    {
        // Nothing to do

    }
}
