/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package org.ametys.plugins.survey.clientsideelement;

import java.text.DateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.i18n.I18nUtils;

import org.ametys.core.ui.Callable;
import org.ametys.core.ui.StaticClientSideElement;
import org.ametys.core.util.DateUtils;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.survey.repository.Survey;
import org.ametys.runtime.i18n.I18nizableText;

/**
 * ClientSideElement for the Scheduled Survey
 */
public class ScheduledSurveyClientSideElement extends StaticClientSideElement implements Contextualizable
{
    private Context _context;
    private AmetysObjectResolver _resolver;

    @Override
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
    }
    
    /**
     * Get the scheduled dates of a survey
     * @param surveyId the id of the survey
     * @return a map
     */
    @Callable(rights = "Plugins_Survey_Right_Handle", context = "/cms")
    public Map<String, Object> getScheduledDates(String surveyId)
    {
        Map<String, Object> dates = new HashMap<> ();
        
        Survey survey = _resolver.resolveById(surveyId);
        
        Date startDate = survey.getStartDate();
        if (startDate != null)
        {
            dates.put("startDate", DateUtils.dateToString(startDate));
        }
        
        Date endDate = survey.getEndDate();
        if (endDate != null)
        {
            dates.put("endDate", DateUtils.dateToString(endDate));
        }
        
        return dates;
    }
    
    /**
     * Set date of publication of surveys
     * @param surveyIds The ids of surveys to update
     * @param startDateAsStr The start date. Can be null.
     * @param endDateAsStr The end date. Can be null.
     * @return true if operation has succeeded.
     */
    @Callable(rights = "Plugins_Survey_Right_Handle", context = "/cms")
    public boolean setScheduledDate (List<String> surveyIds, String startDateAsStr, String endDateAsStr)
    {
        Date startDate = startDateAsStr != null ? DateUtils.parse(startDateAsStr) : null;
        Date endDate = endDateAsStr != null ? DateUtils.parse(endDateAsStr) : null;
        
        for (String id : surveyIds)
        {
            Survey survey = _resolver.resolveById(id);
            
            survey.setStartDate(startDate);
            survey.setEndDate(endDate);
            survey.saveChanges();
        }
        
        return true;
    }
    
    /**
     * Get the survey status
     * @param surveysId The survey ids
     * @return the surveys status
     */
    @SuppressWarnings("unchecked")
    @Callable(rights = "Plugins_Survey_Right_Handle", context = "/cms")
    public Map<String, Object> getStatus (List<String> surveysId)
    {
        Map<String, Object> results = new HashMap<>();
        
        results.put("allright-surveys", new ArrayList<>());
        results.put("scheduled-surveys", new ArrayList<>());
        results.put("scheduled-valid-surveys", new ArrayList<>());
        results.put("scheduled-forthcoming-surveys", new ArrayList<>());
        results.put("scheduled-outofdate-surveys", new ArrayList<>());
        
        Calendar now = new GregorianCalendar();
        now.set(Calendar.HOUR_OF_DAY, 0);
        now.set(Calendar.MINUTE, 0);
        now.set(Calendar.SECOND, 0);
        now.set(Calendar.MILLISECOND, 0);
        
        for (String surveyId : surveysId)
        {
            Survey survey = _resolver.resolveById(surveyId);
            
            Map<String, Object> surveyParams = new HashMap<>();
            surveyParams.put("id", survey.getId());
            surveyParams.put("title", survey.getTitle());
            
            Date startDate = survey.getStartDate();
            Date endDate = survey.getEndDate();
            
            boolean isScheduled = startDate != null || endDate != null;
            if (isScheduled)
            {
                Map objectModel = ContextHelper.getObjectModel(_context);
                Locale locale = I18nUtils.findLocale(objectModel, "locale", null, Locale.getDefault(), true);
                
                if ((startDate == null || !startDate.after(now.getTime())) && (endDate == null || !endDate.before(now.getTime())))
                {
                    surveyParams.put("description", _getDateValidDescription(survey));
                    
                    List<Map<String, Object>> validSurveys = (List<Map<String, Object>>) results.get("scheduled-valid-surveys");
                    validSurveys.add(surveyParams);
                }
                else if (endDate != null && endDate.before(now.getTime()))
                {
                    surveyParams.put("description", _getOutOfDateDescription(survey, endDate, locale));
                    
                    List<Map<String, Object>> oodSurveys = (List<Map<String, Object>>) results.get("scheduled-outofdate-surveys");
                    oodSurveys.add(surveyParams);
                }
                else if (startDate != null && startDate.after(now.getTime()))
                {
                    surveyParams.put("description", _getForthComingDescription(survey, startDate, endDate, locale));
                    
                    List<Map<String, Object>> forthcomingSurveys = (List<Map<String, Object>>) results.get("scheduled-forthcoming-surveys");
                    forthcomingSurveys.add(surveyParams);
                }
                
                List<Map<String, Object>> scheduledSurveys = (List<Map<String, Object>>) results.get("scheduled-surveys");
                scheduledSurveys.add(surveyParams);
            }
            
            List<Map<String, Object>> allrightSurveys = (List<Map<String, Object>>) results.get("allright-surveys");
            allrightSurveys.add(surveyParams);
        }
        
        return results;
    }
    

    private I18nizableText _getDateValidDescription (Survey survey)
    {
        List<String> i18nParameters = new ArrayList<>();
        i18nParameters.add(survey.getTitle());
        
        I18nizableText ed = (I18nizableText) this._script.getParameters().get("scheduled-survey-valid-description");
        return new I18nizableText(ed.getCatalogue(), ed.getKey(), i18nParameters);
    }
    
    private I18nizableText _getOutOfDateDescription (Survey survey, Date endDate, Locale locale)
    {
        List<String> i18nParameters = new ArrayList<>();
        i18nParameters.add(survey.getTitle());
        
        DateFormat df = DateFormat.getDateInstance(DateFormat.LONG, locale);
        i18nParameters.add(endDate != null ? df.format(endDate) : "-");
        
        I18nizableText ed = (I18nizableText) this._script.getParameters().get("scheduled-survey-outofdate-description");
        return new I18nizableText(ed.getCatalogue(), ed.getKey(), i18nParameters);
    }
    
    private I18nizableText _getForthComingDescription (Survey survey, Date startDate, Date endDate, Locale locale)
    {
        List<String> i18nParameters = new ArrayList<>();
        i18nParameters.add(survey.getTitle());
        
        DateFormat df = DateFormat.getDateInstance(DateFormat.LONG, locale);
        i18nParameters.add(startDate != null ? df.format(startDate) : "-");
        i18nParameters.add(endDate != null ? df.format(endDate) : "-");
        
        I18nizableText ed = (I18nizableText) this._script.getParameters().get("scheduled-survey-forthcoming-description");
        return new I18nizableText(ed.getCatalogue(), ed.getKey(), i18nParameters);
    }
    
}
