/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.survey.dao;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.jcr.Node;
import javax.jcr.NodeIterator;
import javax.jcr.Repository;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.query.Query;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang3.StringUtils;

import org.ametys.core.observation.Event;
import org.ametys.core.ui.Callable;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.jcr.NameHelper;
import org.ametys.plugins.repository.provider.AbstractRepository;
import org.ametys.plugins.survey.SurveyEvents;
import org.ametys.plugins.survey.repository.Survey;
import org.ametys.plugins.survey.repository.SurveyPage;
import org.ametys.plugins.survey.repository.SurveyQuestion;
import org.ametys.plugins.survey.repository.SurveyQuestion.QuestionType;
import org.ametys.plugins.survey.repository.SurveyRule;
import org.ametys.plugins.survey.repository.SurveyRule.RuleType;

/**
 * DAO for manipulating survey pages.
 *
 */
public class PageDAO extends AbstractDAO
{
    /** The Avalon role */
    public static final String ROLE = PageDAO.class.getName();
    
    /** The repository */
    private Repository _repository;
    
    /** The Question DAO */
    private QuestionDAO _questionDAO;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _repository = (Repository) serviceManager.lookup(AbstractRepository.ROLE);
        _questionDAO = (QuestionDAO) serviceManager.lookup(QuestionDAO.ROLE);
    }
    
    /**
     * Gets properties of a survey page
     * @param id The id of the survey page
     * @return The properties
     */
    @Callable(rights = "Plugins_Survey_Right_Handle", context = "/cms")
    public Map<String, Object> getPage (String id)
    {
        SurveyPage page = _resolver.resolveById(id);
        
        return getPage(page);
    }
    
    /**
     * Gets properties of a survey page
     * @param page The survey page
     * @return The properties
     */
    public Map<String, Object> getPage (SurveyPage page)
    {
        Map<String, Object> properties = new HashMap<>();
        
        properties.put("id", page.getId());
        properties.put("label", page.getLabel());
        properties.put("title", page.getTitle());
        properties.put("description", page.getDescription());
        
        properties.putAll(getPictureInfo(page));
        
        return properties;
    }
    
    /**
     * Determines if a page is the last of survey's pages.
     * @param id The page id
     * @return True if the page is the last one. 
     */
    @Callable(rights = "Plugins_Survey_Right_Handle", context = "/cms")
    public boolean isLastPage (String id)
    {
        SurveyPage page = _resolver.resolveById(id);
        
        Survey survey = page.getParent();
        AmetysObjectIterable<SurveyPage> pages = survey.getChildren();
        Iterator<SurveyPage> it = pages.iterator();
        
        SurveyPage lastPage = null;
        while (it.hasNext())
        {
            lastPage = it.next();
        }
        
        boolean isLast = lastPage != null && id.equals(lastPage.getId());
        return isLast;
    }
    
    /**
     * Creates a survey page.
     * @param values The survey page's values
     * @return The id of the created survey page
     * @throws Exception if an exception occurs during the page creation process
     */
    @Callable(rights = "Plugins_Survey_Right_Handle", context = "/cms")
    public Map<String, String> createPage (Map<String, Object> values) throws Exception
    {
        Map<String, String> result = new HashMap<>();
        
        String surveyId = StringUtils.defaultString((String) values.get("surveyId"));
        Survey survey = _resolver.resolveById(surveyId);
        
        String label = StringUtils.defaultString((String) values.get("label"));
        String originalName = NameHelper.filterName(label);
        
        // Find unique name
        String name = originalName;
        int index = 2;
        while (survey.hasChild(name))
        {
            name = originalName + "-" + (index++);
        }
        
        SurveyPage page = survey.createChild(name, "ametys:survey-page");
        _setValues(page, values);
        
        survey.saveChanges();
        
        Map<String, Object> eventParams = new HashMap<>();
        eventParams.put("survey", survey);
        _observationManager.notify(new Event(SurveyEvents.SURVEY_MODIFIED, _getCurrentUser(), eventParams));
        
        result.put("id", page.getId());
        
        return result;
    }
    
    /**
     * Edits a survey page.
     * @param values The survey page's values
     * @return The id of the edited survey page and the id of its survey parent
     */
    @Callable(rights = "Plugins_Survey_Right_Handle", context = "/cms")
    public Map<String, String> editPage (Map<String, Object> values)
    {
        Map<String, String> result = new HashMap<>();
        
        String id = StringUtils.defaultString((String) values.get("id"));
        SurveyPage page = _resolver.resolveById(id);
        
        _setValues(page, values);
        
        page.saveChanges();
        
        Map<String, Object> eventParams = new HashMap<>();
        eventParams.put("survey", page.getSurvey());
        _observationManager.notify(new Event(SurveyEvents.SURVEY_MODIFIED, _getCurrentUser(), eventParams));
        
        result.put("id", page.getId());
        result.put("surveyId", page.getSurvey().getId());
        
        return result;
    }
    
    private void _setValues (SurveyPage page, Map<String, Object> values)
    {
        page.setTitle(StringUtils.defaultString((String) values.get("title")));
        page.setLabel(StringUtils.defaultString((String) values.get("label")));
        page.setDescription(StringUtils.defaultString((String) values.get("description")));
        
        page.setPictureAlternative(StringUtils.defaultString((String) values.get("picture-alternative")));
        setPicture(page, StringUtils.defaultString((String) values.get("picture")));
    }
    
    /**
     * Copies and pastes a survey page.
     * @param surveyId The id of the survey, target of the copy
     * @param pageId The id of the page to copy
     * @return The id of the created page
     */
    @Callable(rights = "Plugins_Survey_Right_Handle", context = "/cms")
    public Map<String, String> copyPage(String surveyId, String pageId)
    {
        Map<String, String> result = new HashMap<>();
        
        SurveyPage originalPage = _resolver.resolveById(pageId);
        Survey parentSurvey = _resolver.resolveById(surveyId);
        
        // Find unique name
        String originalName = originalPage.getName();
        String name = originalName;
        int index = 2;
        while (parentSurvey.hasChild(name))
        {
            name = originalName + "-" + (index++);
        }
        
        SurveyPage cPage = originalPage.copyTo(parentSurvey, name);
        
        Survey originalSurvey = originalPage.getSurvey();
        if (!originalSurvey.getId().equals(parentSurvey.getId()))
        {
            // Update rules references after copy
            updateReferencesAfterCopy (originalPage.getSurvey(), parentSurvey, cPage);
        }
        
        parentSurvey.saveChanges();
        
        Map<String, Object> eventParams = new HashMap<>();
        eventParams.put("survey", parentSurvey);
        _observationManager.notify(new Event(SurveyEvents.SURVEY_MODIFIED, _getCurrentUser(), eventParams));
        
        result.put("id", cPage.getId());
        
        return result;
    }
    
    /**
     * Deletes a survey page.
     * @param id The id of the survey page to delete
     * @return The id of the deleted survey page and the id of its survey parent
     */
    @Callable(rights = "Plugins_Survey_Right_Handle", context = "/cms")
    public Map<String, String> deletePage (String id)
    {
        Map<String, String> result = new HashMap<>();
        
        SurveyPage page = _resolver.resolveById(id);
        Survey survey = page.getParent();
        
        page.remove();
        
        // Remove rules references
        _removeReferencesFromPages (id);
        _removeReferencesFromQuestions(id);
        
        survey.saveChanges();
        
        Map<String, Object> eventParams = new HashMap<>();
        eventParams.put("survey", survey);
        _observationManager.notify(new Event(SurveyEvents.SURVEY_MODIFIED, _getCurrentUser(), eventParams));
        
        result.put("id", id);
        result.put("surveyId", survey.getId());
        
        return result;
    }
    
    /**
     * Adds a a new rule to a page.
     * @param id The id of the page
     * @param rule The rule type
     * @param page The page to jump or skip
     * @return An empty map
     */
    @Callable(rights = "Plugins_Survey_Right_Handle", context = "/cms")
    public Map<String, Object> addRule (String id, String rule, String page)
    {
        SurveyPage surveyPage = _resolver.resolveById(id);
        
        surveyPage.setRule(RuleType.valueOf(rule), page);
        surveyPage.saveChanges();
        
        Map<String, Object> eventParams = new HashMap<>();
        eventParams.put("survey", surveyPage.getSurvey());
        _observationManager.notify(new Event(SurveyEvents.SURVEY_MODIFIED, _getCurrentUser(), eventParams));
        
        return new HashMap<>();
    }
    
    /**
     * Deletes a rule to a page
     * @param id The id of the page
     * @return An empty map
     */
    @Callable(rights = "Plugins_Survey_Right_Handle", context = "/cms")
    public Map<String, Object> deleteRule (String id)
    {
        SurveyPage surveyPage = _resolver.resolveById(id);
        
        surveyPage.deleteRule();
        surveyPage.saveChanges();
        
        Map<String, Object> eventParams = new HashMap<>();
        eventParams.put("survey", surveyPage.getSurvey());
        _observationManager.notify(new Event(SurveyEvents.SURVEY_MODIFIED, _getCurrentUser(), eventParams));
        
        return new HashMap<>();
    }
    
    /**
     * Gets the rule for a survey page.
     * @param id The id of the survey page.
     * @return The rule, or null
     */
    @Callable(rights = "Plugins_Survey_Right_Handle", context = "/cms")
    public Map<String, Object> getRule (String id)
    {
        Map<String, Object> result = new HashMap<>();
        
        SurveyPage page = _resolver.resolveById(id);
        SurveyRule rule = page.getRule();
        
        if (rule != null)
        {
            result.put("type", rule.getType().name());
            String pageId = rule.getPage();
            if (pageId != null)
            {
                try
                {
                    SurveyPage pageAO = _resolver.resolveById(pageId);
                    result.put("page", pageId);
                    result.put("pageName", pageAO.getLabel());
                }
                catch (UnknownAmetysObjectException e)
                {
                    // The page does not exist anymore
                }
            }
        }
        else
        {
            result = null;
        }
        
        return result;
    }
    
    /**
     * Gets the branches for a survey page.
     * @param id The id of the survey page.
     * @return The branches
     */
    @Callable(rights = "Plugins_Survey_Right_Handle", context = "/cms")
    public Map<String, Object> getBranches (String id)
    {
        Map<String, Object> result = new HashMap<>();
        
        SurveyPage page = _resolver.resolveById(id);
        
        result.put("id", page.getId());
        
        List<Object> questions = new ArrayList<>();
        AmetysObjectIterable<SurveyQuestion> questionsAO = page.getChildren();
        int index = 1;
        for (SurveyQuestion question : questionsAO)
        {
            if (question.getType() == QuestionType.SINGLE_CHOICE || question.getType() == QuestionType.MULTIPLE_CHOICE)
            {
                questions.add(_questionDAO.getRules(question.getId(), index));
            }
            index++;
        }
        result.put("questions", questions);
        
        // SAX page rule
        result.put("rule", getRule(id));
        
        return result;
    }
    
    private void _removeReferencesFromPages (String pageId)
    {
        Session session = null;
        String jcrQuery = "//element(*, ametys:survey-page)/element(*, ametys:survey-rule)[@ametys-internal:page='" + pageId + "']"; 
        
        try
        {
            session = _repository.login();
            @SuppressWarnings("deprecation")
            Query query = session.getWorkspace().getQueryManager().createQuery(jcrQuery, Query.XPATH);
            
            NodeIterator nodes = query.execute().getNodes();
            while (nodes.hasNext())
            {
                Node ruleNode = nodes.nextNode();
                Node pageNode = ruleNode.getParent();
                
                pageNode.getNode(RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":rule").remove();
                pageNode.getSession().save();
            }
        }
        catch (RepositoryException ex)
        {
            if (session != null)
            {
                session.logout();
            }

            throw new AmetysRepositoryException("An error occurred executing the JCR query : " + jcrQuery, ex);
        }
    }
    
    private void _removeReferencesFromQuestions (String pageId)
    {
        Session session = null;
        String jcrQuery = "//element(*, ametys:survey-question)//element(*, ametys:survey-rule)[@ametys-internal:page='" + pageId + "']"; 
        
        try
        {
            session = _repository.login();
            @SuppressWarnings("deprecation")
            Query query = session.getWorkspace().getQueryManager().createQuery(jcrQuery, Query.XPATH);
            
            NodeIterator nodes = query.execute().getNodes();
            while (nodes.hasNext())
            {
                Node ruleNode = nodes.nextNode();
                Node questionNode = ruleNode.getParent().getParent();
                
                questionNode.getNode(RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":rules").getNode(ruleNode.getName()).remove();
                questionNode.getSession().save();
            }
        }
        catch (RepositoryException ex)
        {
            if (session != null)
            {
                session.logout();
            }

            throw new AmetysRepositoryException("An error occurred executing the JCR query : " + jcrQuery, ex);
        }
    }
}
