/*
 *  Copyright 2011 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.survey.generators;

import java.io.IOException;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.generation.ServiceableGenerator;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.xml.sax.SAXException;

import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.survey.repository.SurveyPage;
import org.ametys.plugins.survey.repository.SurveyQuestion;
import org.ametys.plugins.survey.repository.SurveyRule;

/**
 * SAX rules of a {@link SurveyQuestion}
 *
 */
public class QuestionRulesGenerator extends ServiceableGenerator
{
    /** The Ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) serviceManager.lookup(AmetysObjectResolver.ROLE);
    }
    
    @Override
    public void generate() throws IOException, SAXException, ProcessingException
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        String id = request.getParameter("id");
        
        AmetysObject ao = _resolver.resolveById(id);
        
        contentHandler.startDocument();
        
        if (ao instanceof SurveyQuestion)
        {
            saxRules((SurveyQuestion) ao, 0);
        }
        else if (ao instanceof SurveyPage)
        {
            saxRule ((SurveyPage) ao);
        }
        
        contentHandler.endDocument();
    }
    
    /**
     * SAX rules
     * @param question the question
     * @param number the question number 
     * @throws SAXException if an error occurs while saxing
     */
    public void saxRules (SurveyQuestion question, int number) throws SAXException
    {
        Map<String, String> options = question.getOptions();
        
        AttributesImpl attr = new AttributesImpl();
        attr = new AttributesImpl();
        attr.addCDATAAttribute("id", question.getId());
        attr.addCDATAAttribute("number", String.valueOf(number));
        
        XMLUtils.startElement(contentHandler, "question", attr);
        XMLUtils.createElement(contentHandler, "title", question.getTitle());
        
        XMLUtils.startElement(contentHandler, "rules");
        List<SurveyRule> rules = question.getRules();
        for (SurveyRule rule : rules)
        {
            AttributesImpl atts = new AttributesImpl();
            
            String option = rule.getOption();
            atts.addCDATAAttribute("option", option);
            atts.addCDATAAttribute("optionLabel", options.get(option));
            
            atts.addCDATAAttribute("type", rule.getType().name());
            String page = rule.getPage();
            if (page != null)
            {
                try
                {
                    SurveyPage pageAO = _resolver.resolveById(page);
                    atts.addCDATAAttribute("page", page);
                    atts.addCDATAAttribute("pageName", pageAO.getLabel());
                }
                catch (UnknownAmetysObjectException e)
                {
                    // Page does not exist anymore
                }
            }
            XMLUtils.createElement(contentHandler, "rule", atts);
        }
        XMLUtils.endElement(contentHandler, "rules");
        
        XMLUtils.endElement(contentHandler, "question");
    }
    
    /**
     * SAX page rule
     * @param surveyPage the survey page
     * @throws SAXException if an error occurs while saxing
     */
    public void saxRule (SurveyPage surveyPage) throws SAXException
    {
        SurveyRule rule = surveyPage.getRule();
        
        if (rule != null)
        {
            AttributesImpl atts = new AttributesImpl();
            atts.addCDATAAttribute("type", rule.getType().name());
            String page = rule.getPage();
            if (page != null)
            {
                try
                { 
                    atts.addCDATAAttribute("page", page);
                    SurveyPage pageAO = _resolver.resolveById(page);
                    atts.addCDATAAttribute("pageName", pageAO.getLabel());
                }
                catch (UnknownAmetysObjectException e)
                {
                    // The page does not exist anymore
                }
            }
            
            XMLUtils.createElement(contentHandler, "rule", atts);
        }
    }

}
