/*
 *  Copyright 2011 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.survey.repository;

import java.util.List;

import javax.jcr.Node;
import javax.jcr.PathNotFoundException;
import javax.jcr.RepositoryException;

import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.ModifiableTraversableAmetysObject;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.survey.repository.SurveyRule.RuleType;

/**
 * {@link AmetysObject} for storing survey
 */
public class SurveyPage extends AbstractSurveyElement<SurveyPageFactory>
{
    /** Constants for title metadata. */
    private static final String __PROPERTY_LABEL = RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":label";
    /** Constants for title metadata. */
    private static final String __PROPERTY_TITLE = RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":title";
    /** Constants for description metadata. */
    private static final String __PROPERTY_DESC = RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":description";
    /** Constants for rules metadata. */
    private static final String __PROPERTY_RULE = RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":rule";
    private static final String __PROPERTY_RULE_TYPE = RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":rule";
    private static final String __PROPERTY_RULE_PAGE = RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":page";
    
    /**
     * Creates a {@link SurveyPage}.
     * @param node the node backing this {@link AmetysObject}.
     * @param parentPath the parent path in the Ametys hierarchy.
     * @param factory the {@link SurveyFactory} which creates the AmetysObject.
     */
    public SurveyPage(Node node, String parentPath, SurveyPageFactory factory)
    {
        super(node, parentPath, factory);
    }

    /**
     * Retrieves the title.
     * @return the title.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public String getTitle() throws AmetysRepositoryException
    {
        try
        {
            return getNode().getProperty(__PROPERTY_TITLE).getString();
        }
        catch (PathNotFoundException e)
        {
            return null;
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get title property", e);
        }
    }
    
    /**
     * Set the title.
     * @param title the title.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public void setTitle(String title) throws AmetysRepositoryException
    {
        try
        {
            getNode().setProperty(__PROPERTY_TITLE, title);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to set title property", e);
        }
    }
    
    /**
     * Retrieves the survey name.
     * @return the the survey name.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public String getLabel() throws AmetysRepositoryException
    {
        try
        {
            return getNode().getProperty(__PROPERTY_LABEL).getString();
        }
        catch (PathNotFoundException e)
        {
            return null;
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get label property", e);
        }
    }
    
    /**
     * Set the survey name.
     * @param name the survey name.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public void setLabel(String name) throws AmetysRepositoryException
    {
        try
        {
            getNode().setProperty(__PROPERTY_LABEL, name);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to set label property", e);
        }
    }
    
    /**
     * Retrieves the description.
     * @return the description.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public String getDescription() throws AmetysRepositoryException
    {
        try
        {
            return getNode().getProperty(__PROPERTY_DESC).getString();
        }
        catch (PathNotFoundException e)
        {
            return null;
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get description property", e);
        }
    }
    
    /**
     * Set the description.
     * @param description the description.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public void setDescription(String description) throws AmetysRepositoryException
    {
        try
        {
            getNode().setProperty(__PROPERTY_DESC, description);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to set description property", e);
        }
    }
    
    /**
     * Set the rule for branching
     * @param ruleType the rule type
     * @param page the page to jump or skip. Can be null.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public void setRule (RuleType ruleType, String page) throws AmetysRepositoryException
    {
        try
        {
            if (!getNode().hasNode(__PROPERTY_RULE))
            {
                getNode().addNode(__PROPERTY_RULE, "ametys:survey-rule");
            }
            
            
            Node ruleNode = getNode().getNode(__PROPERTY_RULE);
            
            ruleNode.setProperty(__PROPERTY_RULE_TYPE, ruleType.name());
            if (ruleType == RuleType.JUMP || ruleType == RuleType.SKIP)
            {
                ruleNode.setProperty(__PROPERTY_RULE_PAGE, page);
            }
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to add rule", e);
        }
    }
    
    /**
     * Delete the rule
     * @throws AmetysRepositoryException  if an error occurs.
     */
    public void deleteRule () throws AmetysRepositoryException
    {
        try
        {
            if (getNode().hasNode(__PROPERTY_RULE))
            {
                getNode().getNode(__PROPERTY_RULE).remove();
            }
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to delete rule", e);
        }
    }
    
    /**
     * Determines if the page has a rule
     * @return true if the page has a rule
     * @throws AmetysRepositoryException if an error occurs.
     */
    public boolean hasRule () throws AmetysRepositoryException
    {
        try
        {
            return getNode().hasNode(__PROPERTY_RULE);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get rule property", e);
        }
    }
    
    /**
     * Get rule
     * @return the rules
     * @throws AmetysRepositoryException if an error occurs.
     */
    public SurveyRule getRule () throws AmetysRepositoryException
    {
        try
        {
            if (getNode().hasNode(__PROPERTY_RULE))
            {
                Node node = getNode().getNode(__PROPERTY_RULE);
                RuleType type = RuleType.valueOf(node.getProperty(__PROPERTY_RULE_TYPE).getString());
                String page = null;
                if (node.hasProperty(__PROPERTY_RULE_PAGE))
                {
                    page = node.getProperty(__PROPERTY_RULE_PAGE).getString();
                }
                
                return new SurveyRule(type, page);
            }
            
            return null;
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get rule property", e);
        }
    }
    
    /**
     * Get the Survey to which this page belongs.
     * @return the Survey to which this page belongs.
     * @throws AmetysRepositoryException if an error occurs when retrieving the survey of a page
     */
    public Survey getSurvey() throws AmetysRepositoryException
    {
        return getParent();
    }
    
    /**
     * Determines if question exists
     * @param name The question name
     * @return &lt;code&gt;true&lt;/code&gt; true if the question exists
     * @throws AmetysRepositoryException if an error occurs when looking for a survey question
     */
    public boolean hasQuestion (String name) throws AmetysRepositoryException
    {
        return hasChild(name);
    }
    
    /**
     * Get a question.
     * @param name the question name.
     * @return the question.
     * @throws AmetysRepositoryException if an error occurs when retrieving a question of a survey
     */
    public SurveyQuestion getQuestion(String name) throws AmetysRepositoryException
    {
        return getChild(name);
    }
    
    /**
     * Get the page's questions.
     * @return the page's questions.
     * @throws AmetysRepositoryException if an error occurs when retrieving all the questions of a survey
     */
    public AmetysObjectIterable<SurveyQuestion> getQuestions() throws AmetysRepositoryException
    {
        return getChildren();
    }
    
    @Override
    public SurveyPage copyTo(ModifiableTraversableAmetysObject parent, String name) throws AmetysRepositoryException
    {
        SurveyPage page = parent.createChild(name, "ametys:survey-page");
        page.setTitle(getTitle());
        page.setLabel(getLabel());
        
        String description = getDescription();
        if (description != null)
        {
            page.setDescription(description);
        }
        
        copyPictureTo(page);
        
        SurveyRule rule = getRule();
        if (rule != null)
        {
            page.setRule(getRule().getType(), getRule().getPage());
        }
        
        Survey parentSurvey = page.getSurvey();
        for (SurveyQuestion question : getQuestions())
        {
            question.copyTo(page, parentSurvey.findUniqueQuestionName(question.getName()));
        }
        
        return page;
    }
    
    @Override
    public SurveyPage copyTo(ModifiableTraversableAmetysObject parent, String name, List<String> restrictTo) throws AmetysRepositoryException
    {
        return copyTo(parent, name);
    }
}
