/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.survey.right;

import java.util.Collections;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import javax.jcr.RepositoryException;

import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.components.ContextHelper;

import org.ametys.core.right.AccessController;
import org.ametys.core.right.RightsException;
import org.ametys.plugins.core.impl.right.AbstractHierarchicalAccessController;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.survey.dao.SurveyDAO;
import org.ametys.plugins.survey.repository.AbstractSurveyElement;
import org.ametys.plugins.survey.repository.Survey;
import org.ametys.plugins.survey.repository.SurveyPage;
import org.ametys.plugins.survey.repository.SurveyQuestion;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.i18n.I18nizableTextParameter;
import org.ametys.web.WebHelper;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.site.SiteManager;
import org.ametys.web.repository.sitemap.Sitemap;

/**
 * {@link AccessController} for a {@link Survey}
 */
public class SurveyAccessController extends AbstractHierarchicalAccessController<AmetysObject> implements Contextualizable
{
    /** The site manager */
    protected SiteManager _siteManager;
    /** The survey DAO */
    protected SurveyDAO _surveyDAO;
    private Context _context;

    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _siteManager = (SiteManager) manager.lookup(SiteManager.ROLE);
        _surveyDAO = (SurveyDAO) manager.lookup(SurveyDAO.ROLE);
    }
    
    public boolean supports(Object object)
    {
        return object instanceof AbstractSurveyElement;
    }
    
    @Override
    protected Set<AmetysObject> _getParents(AmetysObject object)
    {
        AmetysObject parent = object.getParent();
        if (supports(parent))
        {
            return Collections.singleton(parent);
        }
        else
        {
            return null;
        }
    }
    
    @Override
    protected boolean ignoreOnHasAnyPermission()
    {
        return true;
    }
    
    @Override
    protected Set< ? extends Object> _convertWorkspaceToRootRightContexts(Set<Object> workspacesContexts)
    {
        String siteName = WebHelper.getSiteName(ContextHelper.getRequest(_context));
        if (siteName != null)
        {
            Site site = _siteManager.getSite(siteName);
            try (AmetysObjectIterable<Sitemap> sitemaps = site.getSitemaps())
            {
                Set<Object> roots = new HashSet<>();
                for (Sitemap sitemap: sitemaps)
                {
                    roots.add(_surveyDAO.getSurveyRootNode(siteName, sitemap.getName()));
                }
                return roots;
            }
            catch (AmetysRepositoryException | RepositoryException e)
            {
                getLogger().error("Failed to convert workspace to root right contexts. The controller will be ignored", e);
                return null;
            }
        }
        return null;
    }
    
    @Override
    protected I18nizableText getObjectLabelForExplanation(Object object) throws RightsException
    {
        Map<String, I18nizableTextParameter> params = Map.of("title", getObjectLabel(object));
        return new I18nizableText("plugin.survey", "PLUGINS_SURVEY_ACCESS_CONTROLLER_CONTEXT_LABEL", params);
    }
    
    public I18nizableText getObjectLabel(Object object) throws RightsException
    {
        if (object instanceof AbstractSurveyElement element)
        {
            Survey survey = null;
            if (element instanceof SurveyPage page)
            {
                survey = page.getSurvey();
            }
            else if (element instanceof Survey s)
            {
                survey = s;
            }
            else if (element instanceof SurveyQuestion q)
            {
                survey = q.getSurvey();
            }
            
            if (survey != null)
            {
                return new I18nizableText(survey.getLabel());
            }
        }
        throw new RightsException("Unsupported object " + object.toString());
    }
    
    public I18nizableText getObjectCategory(Object object)
    {
        return new I18nizableText("plugin.survey", "PLUGINS_SURVEY_ACCESS_CONTROLLER_CONTEXT_CATEGORY");
    }
}
