/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.survey.cache;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.core.right.RightManager;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.data.holder.ModelAwareDataHolder;
import org.ametys.plugins.survey.dao.SurveyDAO;
import org.ametys.plugins.survey.repository.Survey;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.page.ZoneItem;
import org.ametys.web.service.StaticService;

/**
 * Service for displaying a survey, override StaticService to make it cacheable only if survey is public.
 */
public class SurveyService extends StaticService
{

    /** The Ametys object resolver */
    protected AmetysObjectResolver _resolver;

    /** The right manager */
    protected RightManager _rightManager;

    /** The survey DAO */
    protected SurveyDAO _surveyDAO;

    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _rightManager = (RightManager) manager.lookup(RightManager.ROLE);
        _surveyDAO = (SurveyDAO) manager.lookup(SurveyDAO.ROLE);
    }

    @Override
    public boolean isCacheable(Page currentPage, ZoneItem zoneItem)
    {
        ModelAwareDataHolder serviceParameters = zoneItem.getServiceParameters();
        String surveyId = serviceParameters.getValue("surveyId");
        Survey survey = _resolver.resolveById(surveyId);

        // The survey must be public for the service to be cacheable
        return !_surveyDAO.isPrivate(survey);
    }

}
