/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.survey.repository;

import java.io.IOException;
import java.io.InputStream;
import java.time.ZonedDateTime;

import javax.jcr.Node;
import javax.jcr.PathNotFoundException;
import javax.jcr.RepositoryException;

import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.data.Binary;
import org.ametys.cms.data.type.ModelItemTypeConstants;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.CopiableAmetysObject;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.repository.data.ametysobject.ModifiableModelLessDataAwareAmetysObject;
import org.ametys.plugins.repository.data.holder.ModifiableModelLessDataHolder;
import org.ametys.plugins.repository.data.holder.impl.DefaultModifiableModelLessDataHolder;
import org.ametys.plugins.repository.data.repositorydata.impl.JCRRepositoryData;
import org.ametys.plugins.repository.jcr.DefaultTraversableAmetysObject;
import org.ametys.plugins.repository.jcr.DefaultTraversableAmetysObjectFactory;

/**
 * Abstract {@link AmetysObject} for storing elements of a survey
 * @param <F> the actual type of factory.
 */
public abstract class AbstractSurveyElement<F extends SurveyElementFactory> extends DefaultTraversableAmetysObject<F> implements CopiableAmetysObject, ModifiableModelLessDataAwareAmetysObject
{
    /** Constant for picture data property. */
    public static final String PROPERTY_PICTURE = "picture";
    /** Constant for picture type property. */
    public static final String PROPERTY_PICTURE_TYPE = RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":picture-type";
    /** Constant for picture id property. */
    public static final String PROPERTY_PICTURE_ID = RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":picture-id";
    /** Constant for picture alternative property. */
    public static final String PROPERTY_PICTURE_ALTERNATIVE = RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":picture-alternative";
    
    /**
     * Constructor
     * @param node the node backing this {@link AmetysObject}.
     * @param parentPath the parent path in the Ametys hierarchy.
     * @param factory the {@link DefaultTraversableAmetysObjectFactory} which creates the AmetysObject.
     */
    public AbstractSurveyElement(Node node, String parentPath, F factory)
    {
        super(node, parentPath, factory);
    }

    /**
     * Get the picture as a binary metadata.
     * @return the picture as a binary metadata.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public Binary getExternalPicture() throws AmetysRepositoryException
    {
        return getValue(PROPERTY_PICTURE);
    }
    
    /**
     * Set the picture from an external file.
     * @param mimeType the file MIME type.
     * @param filename the file name.
     * @param stream an input stream on the file bytes.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public void setExternalPicture(String mimeType, String filename, InputStream stream) throws AmetysRepositoryException
    {
        try
        {
            removePictureMetas();
            
            setPictureType("external");
            
            Binary pic = new Binary();
            
            pic.setLastModificationDate(ZonedDateTime.now());
            pic.setInputStream(stream);
            pic.setMimeType(mimeType);
            pic.setFilename(filename);
            
            setValue(PROPERTY_PICTURE, pic, ModelItemTypeConstants.BINARY_ELEMENT_TYPE_ID);
        }
        catch (IOException | RepositoryException e)
        {
            throw new AmetysRepositoryException("Error setting the external picture property.", e);
        }
    }
    
    /**
     * Get the picture resource ID.
     * @return the resource ID.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public String getResourcePictureId() throws AmetysRepositoryException
    {
        try
        {
            return getNode().getProperty(PROPERTY_PICTURE_ID).getString();
        }
        catch (PathNotFoundException e)
        {
            return null;
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error getting the picture ID property.", e);
        }
    }
        
    /**
     * Set the picture from an explorer resource.
     * @param resourceId the resource ID.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public void setResourcePicture(String resourceId) throws AmetysRepositoryException
    {
        try
        {
            removePictureMetas();
            
            setPictureType("resource");
            
            getNode().setProperty(PROPERTY_PICTURE_ID, resourceId);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error setting the alternative property.", e);
        }
    }
    
    /**
     * Removes any picture currently assigned.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public void setNoPicture() throws AmetysRepositoryException
    {
        try
        {
            setPictureType("");
            
            removePictureMetas();
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error setting the alternative property.", e);
        }
    }

    /**
     * Get the picture type.
     * @return the picture type.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public String getPictureType() throws AmetysRepositoryException
    {
        try
        {
            return getNode().getProperty(PROPERTY_PICTURE_TYPE).getString();
        }
        catch (PathNotFoundException e)
        {
            return null;
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error getting the type property.", e);
        }
    }
    
    /**
     * Set the picture type.
     * @param type the picture type to set.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public void setPictureType(String type) throws AmetysRepositoryException
    {
        try
        {
            getNode().setProperty(PROPERTY_PICTURE_TYPE, type);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error setting the type property.", e);
        }
    }
    
    /**
     * Get the picture alternative.
     * @return the picture alternative.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public String getPictureAlternative() throws AmetysRepositoryException
    {
        try
        {
            return getNode().getProperty(PROPERTY_PICTURE_ALTERNATIVE).getString();
        }
        catch (PathNotFoundException e)
        {
            return null;
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error getting the alternative property.", e);
        }
    }
    
    /**
     * Set the picture alternative.
     * @param alternative the picture alternative to set.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public void setPictureAlternative(String alternative) throws AmetysRepositoryException
    {
        try
        {
            getNode().setProperty(PROPERTY_PICTURE_ALTERNATIVE, alternative);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Error setting the alternative property.", e);
        }
    }
    
    /**
     * Remove all picture metas (picture ID and picture binary).
     * @throws RepositoryException if an error occurs.
     */
    protected void removePictureMetas() throws RepositoryException
    {
        getNode().setProperty(PROPERTY_PICTURE_ID, StringUtils.EMPTY);
        removeValue(PROPERTY_PICTURE);
    }
    
    /**
     * Copy the picture of this {@link AbstractSurveyElement} to the element in arguments
     * @param elmt The element
     */
    protected void copyPictureTo (AbstractSurveyElement elmt)
    {
        String pictureType = getPictureType();
        if (StringUtils.isBlank(pictureType))
        {
            elmt.setNoPicture();
        }
        else if (pictureType.equals("resource"))
        {
            String pictureId = getResourcePictureId();
            if (StringUtils.isNotBlank(pictureId))
            {
                elmt.setResourcePicture(pictureId);
            }
            else
            {
                elmt.setNoPicture();
            }
        }
        else if (pictureType.equals("external"))
        {
            Binary externalPicture = getExternalPicture();
            elmt.setExternalPicture(externalPicture.getMimeType(), externalPicture.getFilename(), externalPicture.getInputStream());
        }
        
        String pictureAlternative = getPictureAlternative();
        if (pictureAlternative != null)
        {
            elmt.setPictureAlternative(pictureAlternative);
        }
    }
    
    public ModifiableModelLessDataHolder getDataHolder()
    {
        JCRRepositoryData repositoryData = new JCRRepositoryData(getNode());
        return new DefaultModifiableModelLessDataHolder(_getFactory().getSurveyElementDataTypeExtensionPoint(), repositoryData);
    }
}
