/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.survey;

import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.util.Calendar;
import java.util.GregorianCalendar;

import org.ametys.core.util.DateUtils;

/**
 * Survey date utils
 * FIXME When the plugins Survey will be not compatible with 4.2.x version, use the methods in {@link DateUtils}
 * See issue SURVEY-208
 */
public final class SurveyDateUtils
{
    private SurveyDateUtils()
    {
        // empty
    }
    
    /** 
     * Converts this {@link Calendar} object to a {@link ZonedDateTime}.
     * @param calendar the calendar
     * @return an instant representing the same point on the time-line as this {@link Calendar} object
     */
    public static ZonedDateTime asZonedDateTime(Calendar calendar)
    {
        ZonedDateTime zonedDateTime =  ZonedDateTime.of(calendar.get(Calendar.YEAR),
                calendar.get(Calendar.MONTH) + 1,
                calendar.get(Calendar.DAY_OF_MONTH),
                calendar.get(Calendar.HOUR_OF_DAY),
                calendar.get(Calendar.MINUTE),
                calendar.get(Calendar.SECOND),
                calendar.get(Calendar.MILLISECOND),
                calendar.getTimeZone().toZoneId());
        return zonedDateTime.withZoneSameInstant(ZoneId.of("UTC"));
    }
    
    /** 
     * Converts this {@link ZonedDateTime} object to a {@link Calendar}.
     * @param zonedDateTime the zoned date time
     * @return the {@link Calendar} object
     */
    public static Calendar asCalendar(ZonedDateTime zonedDateTime)
    {
        ZonedDateTime dateTimeOnUTC = zonedDateTime.withZoneSameInstant(ZoneId.of("UTC"));
        return GregorianCalendar.from(dateTimeOnUTC);
    }
    
    /** 
     * Converts a {@link ZonedDateTime} object to {@link String} using the ISO date formatter
     * @param zonedDateTime the zoned date time
     * @return the zoned date time as a {@link String}
     */
    public static String zonedDateTimeToString(ZonedDateTime zonedDateTime)
    {
        return zonedDateTime.format(DateUtils.getISODateTimeFormatter());
    }
    
    /** 
     * Parses a String into a {@link ZonedDateTime}, using ISO date formatter.
     * @param zonedDateTimeAsString the zoned date time as string
     * @return the {@link ZonedDateTime} object
     */
    public static ZonedDateTime parseZonedDateTime(String zonedDateTimeAsString)
    {
        return ZonedDateTime.parse(zonedDateTimeAsString, DateUtils.getISODateTimeFormatter());
    }
}
