/*
 *  Copyright 2011 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.survey.answer;

import java.net.URI;
import java.net.URISyntaxException;
import java.time.LocalDate;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.format.FormatStyle;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.regex.Pattern;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.acting.ServiceableAction;
import org.apache.cocoon.environment.Cookie;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.Response;
import org.apache.cocoon.environment.SourceResolver;
import org.apache.cocoon.environment.http.HttpCookie;
import org.apache.commons.lang3.StringUtils;

import org.ametys.core.right.RightManager;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.util.DateUtils;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.survey.SurveyDateUtils;
import org.ametys.plugins.survey.data.SurveyAnswer;
import org.ametys.plugins.survey.data.SurveyAnswerDao;
import org.ametys.plugins.survey.data.SurveySession;
import org.ametys.plugins.survey.repository.Survey;
import org.ametys.plugins.survey.repository.SurveyAccessHelper;
import org.ametys.plugins.survey.repository.SurveyPage;
import org.ametys.plugins.survey.repository.SurveyQuestion;
import org.ametys.plugins.survey.repository.SurveyRule;
import org.ametys.runtime.i18n.I18nizableTextParameter;
import org.ametys.runtime.i18n.I18nizableDate;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.web.URIPrefixHandler;
import org.ametys.web.repository.page.Page;

/**
 * Process the user answers to the survey.
 */
public class ProcessInputAction extends ServiceableAction
{
    
    /** The name of the cookie storing the taken surveys. */
    public static final String COOKIE_NAME = "org.ametys.survey.answeredSurveys";
    
    /** The ametys object resolver. */
    protected AmetysObjectResolver _resolver;
    /** The ametys object resolver. */
    protected SurveyAnswerDao _answerDao;
    /** The user provider. */
    protected CurrentUserProvider _userProvider;
    /** The uri prefix handler. */
    protected URIPrefixHandler _prefixHandler;
    /** The survey access helper */
    protected SurveyAccessHelper _accessHelper;
    
    /** The plugin name */
    protected String _pluginName;

    private RightManager _rightManager;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _resolver = (AmetysObjectResolver) serviceManager.lookup(AmetysObjectResolver.ROLE);
        _answerDao = (SurveyAnswerDao) serviceManager.lookup(SurveyAnswerDao.ROLE);
        _userProvider = (CurrentUserProvider) serviceManager.lookup(CurrentUserProvider.ROLE);
        _prefixHandler = (URIPrefixHandler) serviceManager.lookup(URIPrefixHandler.ROLE);
        _accessHelper = (SurveyAccessHelper) serviceManager.lookup(SurveyAccessHelper.ROLE);
        _rightManager = (RightManager) serviceManager.lookup(RightManager.ROLE);
    }
    
    @Override
    public Map act(Redirector redirector, SourceResolver resolver, Map objectModel, String source, Parameters parameters) throws Exception
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        Response response = ObjectModelHelper.getResponse(objectModel);
        
        _pluginName = getPluginName(request);
        
        String surveyId = request.getParameter("surveyId");
        if (StringUtils.isNotEmpty(surveyId))
        {
            // Get the survey object.
            Survey survey = _resolver.resolveById(surveyId);
            
            SurveyErrors errors = new SurveyErrors();
            
            if (!checkAccess(survey, request, errors))
            {
                request.setAttribute("survey", survey);
                request.setAttribute("survey-errors", errors);
                return null;
            }
            
            // Get the user input.
            SurveyInput surveyInput = getInput(survey, request);
            
            // Validate the user input.
            validateInput(survey, surveyInput, errors, request);
            
            // If there were errors in the input, store it as a request attribute and stop.
            if (errors.hasErrors())
            {
                request.setAttribute("survey", survey);
                request.setAttribute("survey-errors", errors);
                return null;
            }
            
            // Add the user session into the database.
            _answerDao.addSession(surveyInput);
            
            setCookie(request, response, surveyId);
            
            // Redirect if necessary.
            String redirectPageId = survey.getRedirection();
            if (StringUtils.isNotEmpty(redirectPageId))
            {
                try
                {
                    Page page = _resolver.resolveById(redirectPageId);
                    redirector.globalRedirect(false, _prefixHandler.getAbsoluteUriPrefix(page.getSiteName()) + "/" + page.getSitemapName() + "/" + page.getPathInSitemap() + ".html");
                }
                catch (UnknownAmetysObjectException e)
                {
                    getLogger().warn("The survey '" + survey.getId() + "' wants to redirect to the unexisting page '" + redirectPageId + "'. Redirecting to default page.", e);
                }
            }
        }
        
        return EMPTY_MAP;
    }
    
    /**
     * Get the plugin name
     * @param request The request
     * @return The plugin name
     */
    protected String getPluginName (Request request)
    {
        return (String) request.getAttribute("pluginName");
    }
    
    /**
     * Check if user can answer to the survey
     * @param survey The survey
     * @param request The request
     * @param errors The survey errors
     * @return false if the access failed
     */
    protected boolean checkAccess(Survey survey, Request request, SurveyErrors errors)
    {
        UserIdentity user = getAuthenticatedUser(request);
        
        if (!_rightManager.hasReadAccess(user, survey))
        {
            // User is not authorized
            errors.addErrors("survey-access", Collections.singletonList(new I18nizableText("plugin." + _pluginName, "PLUGINS_SURVEY_RENDER_UNAUTHORIZED")));
            return false;
        }
        
        String surveyId = survey.getId();
        Date submissionDate = _accessHelper.getSubmissionDate(surveyId, user);
        
        if (submissionDate != null)
        {
            LocalDate localDate = DateUtils.asLocalDate(submissionDate);
            // The authenticated user has already answered to the survey
            Map<String, I18nizableTextParameter> i18nParams = new HashMap<>();
            i18nParams.put("date", new I18nizableDate(localDate, FormatStyle.MEDIUM));
            errors.addErrors("survey-access", Collections.singletonList(new I18nizableText("plugin." + _pluginName, "PLUGINS_SURVEY_RENDER_ALREADY_TAKEN_ON", i18nParams)));
            return false;
        }
        
        // Finally check cookies
        if (user == null && _accessHelper.getCookieName(request, survey) != null)
        {
            // Anonymous user has already answered to the survey
            errors.addErrors("survey-access", Collections.singletonList(new I18nizableText("plugin." + _pluginName, "PLUGINS_SURVEY_RENDER_ALREADY_TAKEN")));
            return false;
        }
        
        // User can access to the survey
        return true;
    }
    
    /**
     * Get the authenticated user
     * @param request The request
     * @return The authenticated user
     */
    protected UserIdentity getAuthenticatedUser (Request request)
    {
        return _userProvider.getUser();
    }
    
    /**
     * Get the user input.
     * @param survey the survey.
     * @param request the request.
     * @return the user input.
     */
    protected SurveyInput getInput(Survey survey, Request request)
    {
        String clientIp = getClientIp(request);
        UserIdentity user = getAuthenticatedUser(request);
        
        SurveyInput surveySession = new SurveyInput();
        
        List<SurveyInputAnswer> answers = new ArrayList<>();
        
        surveySession.setSurveyId(survey.getId());
        surveySession.setSubmittedAt(new Date());
        surveySession.setUser(user);
        surveySession.setIpAddress(clientIp);
        surveySession.setAnswerList(answers);
        
        try (AmetysObjectIterable<SurveyQuestion> questions = survey.getQuestions())
        {
            for (SurveyQuestion question : questions)
            {
                Map<String, Set<String>> values = getValues(question, request);
                
                SurveyInputAnswer answer = new SurveyInputAnswer(question, values);
                answers.add(answer);
            }
            
            return surveySession;
        }
    }
    
    /**
     * Get an answer value from the request.
     * @param question the corresponding question.
     * @param request the request.
     * @return the answer value.
     */
    protected Map<String, Set<String>> getValues(SurveyQuestion question, Request request)
    {
        Map<String, Set<String>> values = new LinkedHashMap<>();
        
        String name = question.getName();
        switch (question.getType())
        {
            case SINGLE_MATRIX:
            case MULTIPLE_MATRIX:
                Collection<String> options = question.getOptions().keySet();
                
                for (String option : options)
                {
                    String paramName = name + "_" + option;
                    String[] paramValues = request.getParameterValues(paramName);
                    
                    if (paramValues != null)
                    {
                        values.put(option, new HashSet<>(Arrays.asList(paramValues)));
                    }
                }
                break;
            case FREE_TEXT:
            case MULTILINE_FREE_TEXT:
                String[] textValues = request.getParameterValues(name);
                if (textValues != null)
                {
                    values.put("values", new HashSet<>(Arrays.asList(textValues)));
                }
                break;
            case SINGLE_CHOICE:
            case MULTIPLE_CHOICE:
            default:
                List<String> valuesAsList = new ArrayList<>();
                String[] paramValues = request.getParameterValues(name);
                if (paramValues != null)
                {
                    for (String value : paramValues)
                    {
                        if (value.equals("__internal_other"))
                        {
                            valuesAsList.add(request.getParameter("__internal_other_" + name));
                        }
                        else
                        {
                            valuesAsList.add(value);
                        }
                    }
                    values.put("values", new HashSet<>(valuesAsList));
                }
                break;
        }
        
        return values;
    }
    
    /**
     * Validate the user input.
     * @param survey the survey.
     * @param input the user input.
     * @param errors the errors.
     * @param request the request.
     */
    protected void validateInput(Survey survey, SurveyInput input, SurveyErrors errors, Request request)
    {
        SurveyRule ruleToExecute = null;
        
        Map<String, SurveyInputAnswer> answers = input.getAnswerMap();
        
        for (SurveyPage page : survey.getPages())
        {
            if (ruleToExecute == null || processPage(page, ruleToExecute))
            {
                // Reset the current rule.
                ruleToExecute = null;
                
                AmetysObjectIterable<SurveyQuestion> questions = page.getQuestions();
                
                for (SurveyQuestion question : questions)
                {
                    SurveyInputAnswer answer = answers.get(question.getName());
                    
                    switch (question.getType())
                    {
                        case FREE_TEXT:
                        case MULTILINE_FREE_TEXT:
                            errors.addErrors(question.getName(), validateText(answer, request));
                            ruleToExecute = evaluateTextRules(question, answer);
                            break;
                        case SINGLE_CHOICE:
                        case MULTIPLE_CHOICE:
                            errors.addErrors(question.getName(), validateChoice(answer, request));
                            ruleToExecute = evaluateChoiceRules(question, answer);
                            break;
                        case SINGLE_MATRIX:
                        case MULTIPLE_MATRIX:
                            errors.addErrors(question.getName(), validateMatrix(answer, request));
                            ruleToExecute = evaluateMatrixRules(question, answer);
                            break;
                        default:
                            break;
                    }
                }
                
                SurveyRule pageRule = page.getRule();
                
                if (ruleToExecute == null && pageRule != null)
                {
                    ruleToExecute = pageRule;
                }
            }
        }
    }
    
    /**
     * Test if a page is to be processed, depending on the rule.
     * @param page the page to test.
     * @param rule the rule to execute.
     * @return true to process the page, false otherwise.
     */
    protected boolean processPage(SurveyPage page, SurveyRule rule)
    {
        boolean processPage = false;
        
        switch (rule.getType())
        {
            case JUMP:
                // If the page is the targeted page, it passes the condition.
                processPage = page.getId().equals(rule.getPage());
                break;
            case SKIP:
                // If the page is the skipped page, it is not displayed.
                processPage = !page.getId().equals(rule.getPage());
                break;
            case FINISH:
                // When finished, no more page is displayed.
                break;
            default:
                break;
        }
        
        return processPage;
    }
    
    /**
     * Evaluate rules on a text question.
     * @param question the text question.
     * @param answer the user answer to the question.
     * @return the matched rule.
     */
    protected SurveyRule evaluateTextRules(SurveyQuestion question, SurveyInputAnswer answer)
    {
        return null;
    }
    
    /**
     * Evaluate rules on a choice question.
     * @param question the choice question.
     * @param answer the user answer to the question.
     * @return the matched rule.
     */
    protected SurveyRule evaluateChoiceRules(SurveyQuestion question, SurveyInputAnswer answer)
    {
        SurveyRule matchedRule = null;
        
        Map<String, Set<String>> valueMap = answer.getValuesMap();
        if (valueMap.containsKey("values"))
        {
            Set<String> values = answer.getValuesMap().get("values");
            
            Iterator<SurveyRule> questionRules = question.getRules().iterator();
            while (questionRules.hasNext() && matchedRule == null)
            {
                SurveyRule rule = questionRules.next();
                
                if (values.contains(rule.getOption()))
                {
                    // Condition met, store the action.
                    matchedRule = rule;
                }
            }
        }
        
        return matchedRule;
    }
    
    /**
     * Evaluate rules on a matrix question.
     * @param question the matrix question.
     * @param answer the user answer to the question.
     * @return the matched rule.
     */
    protected SurveyRule evaluateMatrixRules(SurveyQuestion question, SurveyInputAnswer answer)
    {
        return null;
    }
    
    /**
     * Validate a text field.
     * @param answer the user answer to the question.
     * @param request the request.
     * @return the error list.
     */
    protected List<I18nizableText> validateText(SurveyInputAnswer answer, Request request)
    {
        List<I18nizableText> errors = new ArrayList<>();
        
        SurveyQuestion question = answer.getQuestion();
        
        boolean isBlank = isBlank(answer);
        
        final String textPrefix = "PLUGINS_SURVEY_RENDER_ERROR_TEXT_";
        
        if (question.isMandatory() && isBlank)
        {
            errors.add(new I18nizableText("plugin." + _pluginName, textPrefix + "MANDATORY"));
        }
        
        if (!isBlank)
        {
            errors.addAll(validatePattern(answer, textPrefix));
        }
        
        return errors;
    }
    
    /**
     * Validate a choice question answer.
     * @param answer the user answer to the question.
     * @param request the request.
     * @return the error list.
     */
    protected List<I18nizableText> validateChoice(SurveyInputAnswer answer, Request request)
    {
        List<I18nizableText> errors = new ArrayList<>();
        
        SurveyQuestion question = answer.getQuestion();
        
        boolean isBlank = isBlank(answer);
        
        final String textPrefix = "PLUGINS_SURVEY_RENDER_ERROR_CHOICE_";
        
        if (question.isMandatory() && isBlank)
        {
            errors.add(new I18nizableText("plugin." + _pluginName, textPrefix + "MANDATORY"));
        }
        
        return errors;
    }
    
    /**
     * Validate a matrix question answer.
     * @param answer the user answer to the question.
     * @param request the request.
     * @return the error list.
     */
    protected List<I18nizableText> validateMatrix(SurveyInputAnswer answer, Request request)
    {
        List<I18nizableText> errors = new ArrayList<>();
        
        SurveyQuestion question = answer.getQuestion();
        
        boolean isBlank = isBlank(answer);
        
        final String textPrefix = "PLUGINS_SURVEY_RENDER_ERROR_MATRIX_";
        
        if (question.isMandatory() && isBlank)
        {
            errors.add(new I18nizableText("plugin." + _pluginName, textPrefix + "MANDATORY"));
        }
        
        return errors;
    }
    
    /**
     * Validate an answer against a pattern.
     * @param answer the user answer to the question.
     * @param keyPrefix the error i18n key prefix.
     * @return the error list.
     */
    protected List<I18nizableText> validatePattern(SurveyInputAnswer answer, String keyPrefix)
    {
        List<I18nizableText> errors = new ArrayList<>();
        
        SurveyQuestion question = answer.getQuestion();
        String regex = question.getRegExpPattern();
        String value = answer.getValue();
        
        if (StringUtils.isNotBlank(regex))
        {
            if (!Pattern.matches(regex, value))
            {
                errors.add(new I18nizableText("plugin." + _pluginName, keyPrefix + "PATTERN"));
            }
        }
        
        return errors;
    }
    
    /**
     * Test if the answer is empty.
     * @param answer the user answer.
     * @return true if the answer is empty.
     */
    protected boolean isBlank(SurveyInputAnswer answer)
    {
        boolean blank = answer.getValuesMap().isEmpty();
        
        for (Set<String> values : answer.getValuesMap().values())
        {
            if (StringUtils.isEmpty(StringUtils.join(values, "")))
            {
                return true;
            }
        }
        
        return blank;
    }
    
    /**
     * Indicate in a cookie that the survey was taken.
     * @param request the request.
     * @param response the response.
     * @param surveyId the ID of the survey that was just taken.
     */
    protected void setCookie(Request request, Response response, String surveyId)
    {
        Map<String, Cookie> cookieMap = request.getCookieMap();
        
        Cookie cookie = null;
        String cookieValue = surveyId + "#" + SurveyDateUtils.zonedDateTimeToString(ZonedDateTime.now(ZoneId.of("UTC")));
        if (cookieMap.containsKey(COOKIE_NAME))
        {
            cookie = cookieMap.get(COOKIE_NAME);
            String newValue = cookie.getValue() + '|' + cookieValue;
            cookie.setValue(newValue);
        }
        else
        {
            cookie = response.createCookie(COOKIE_NAME, cookieValue); 
            cookie.setSecure(request.isSecure());
            ((HttpCookie) cookie).getServletCookie().setHttpOnly(false); // Cookie is also manipulated by the javascript
        }
        
        cookie.setVersion(1);
        cookie.setMaxAge(30 * 24 * 3600); // 30 days
        
        String absPrefix = _prefixHandler.getAbsoluteUriPrefix();
        
        String host = null;
        
        try
        {
            URI frontUri = new URI(absPrefix);
            host = frontUri.getHost();
            String path = frontUri.getPath();
            cookie.setPath(StringUtils.isEmpty(path) ? "/" : path);
        }
        catch (URISyntaxException e)
        {
            getLogger().warn("The front URI seems to be invalid.", e);
        }
        
        if (StringUtils.isNotEmpty(host))
        {
            cookie.setDomain(host);
            response.addCookie(cookie);
        }
    }
    
    /**
     * Get a forwarded client IP address if available.
     * @param request The servlet request object.
     * @return The HTTP <code>X-Forwarded-For</code> header value if present,
     * or the default remote address if not.
     */
    protected final String getClientIp(final Request request)
    {
        String ip = "";
        
        String forwardedIpHeader = request.getHeader("X-Forwarded-For");
        
        if (StringUtils.isNotEmpty(forwardedIpHeader))
        {
            String[] forwardedIps = forwardedIpHeader.split("[\\s,]+");
            String forwardedIp = forwardedIps[forwardedIps.length - 1];
            if (StringUtils.isNotBlank(forwardedIp))
            {
                ip = forwardedIp.trim();
            }
        }
        
        if (StringUtils.isBlank(ip))
        {
            ip = request.getRemoteAddr();
        }
        
        return ip;
    }
    
    /**
     * Survey session with answers.
     */
    public class SurveyInput extends SurveySession
    {
        /** Answers. */
        protected List<SurveyInputAnswer> _inputAnswers;
        
        @Override
        public List<SurveyInputAnswer> getAnswers()
        {
            return _inputAnswers;
        }
        
        /**
         * Get the answers as a Map indexed by question ID.
         * @return the answer Map.
         */
        public Map<String, SurveyInputAnswer> getAnswerMap()
        {
            Map<String, SurveyInputAnswer> answerMap = new LinkedHashMap<>();
            
            for (SurveyInputAnswer answer : _inputAnswers)
            {
                answerMap.put(answer.getQuestionId(), answer);
            }
            
            return answerMap;
        }
        
        /**
         * Set the answers.
         * @param answers the answers to set
         */
        public void setAnswerList(List<SurveyInputAnswer> answers)
        {
            this._inputAnswers = answers;
        }
    }
    
    /**
     * Class representing a survey answer, i.e. the response of a user to a question of the survey.
     */
    public class SurveyInputAnswer extends SurveyAnswer
    {
        
        /** The question. */
        protected SurveyQuestion _question;
        
        /** The answer values. */
        protected Map<String, Set<String>> _values;
        
        /**
         * Build a SurveyAnswer object.
         */
        public SurveyInputAnswer()
        {
            this(null, null);
        }
        
        /**
         * Build a SurveyAnswer object.
         * @param question the question ID.
         * @param values the answer value.
         */
        public SurveyInputAnswer(SurveyQuestion question, Map<String, Set<String>> values)
        {
            this._question = question;
            this._values = values;
        }
        
        /**
         * Get the question.
         * @return the question
         */
        public SurveyQuestion getQuestion()
        {
            return _question;
        }
        
        /**
         * Set the question.
         * @param question the question to set
         */
        public void setQuestion(SurveyQuestion question)
        {
            this._question = question;
        }
        
        @Override
        public String getQuestionId()
        {
            return _question.getName();
        }
        
        @Override
        public void setQuestionId(String questionId)
        {
            throw new IllegalAccessError("Set the question instead of the question ID.");
        }
        
        /**
         * Get the values.
         * @return the values
         */
        public Map<String, Set<String>> getValuesMap()
        {
            return _values;
        }
        
        /**
         * Set the values.
         * @param values the values to set
         */
        public void setValueMap(Map<String, Set<String>> values)
        {
            this._values = values;
        }
        
        @Override
        public String getValue()
        {
            String value = "";
            
            switch (_question.getType())
            {
                case SINGLE_MATRIX:
                case MULTIPLE_MATRIX:
                    StringBuilder valueBuff = new StringBuilder();
                    
                    for (String option : _values.keySet())
                    {
                        Set<String> values = _values.get(option);
                        
                        if (valueBuff.length() > 0)
                        {
                            valueBuff.append(";");
                        }
                        valueBuff.append(option).append(":").append(StringUtils.join(values, ","));
                    }
                    
                    value = valueBuff.toString();
                    break;
                case FREE_TEXT:
                case MULTILINE_FREE_TEXT:
                case SINGLE_CHOICE:
                case MULTIPLE_CHOICE:
                default:
                    value = StringUtils.defaultString(StringUtils.join(_values.get("values"), ","));
                    break;
            }
            
            return value;
        }
        
        @Override
        public void setValue(String value)
        {
            throw new IllegalAccessError("Set the value map instead of the vlaue.");
        }
        
    }

}
