/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.survey.dao;

import java.util.HashMap;
import java.util.Map;

import javax.jcr.RepositoryException;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.Constants;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.util.log.SLF4JLoggerAdapter;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.LoggerFactory;

import org.ametys.cms.data.Binary;
import org.ametys.core.observation.ObservationManager;
import org.ametys.core.upload.Upload;
import org.ametys.core.upload.UploadManager;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.util.JSONUtils;
import org.ametys.plugins.explorer.resources.Resource;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.ModifiableTraversableAmetysObject;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.jcr.JCRAmetysObject;
import org.ametys.plugins.survey.repository.AbstractSurveyElement;
import org.ametys.plugins.survey.repository.Survey;
import org.ametys.plugins.survey.repository.SurveyPage;
import org.ametys.plugins.survey.repository.SurveyQuestion;
import org.ametys.plugins.survey.repository.SurveyRule;
import org.ametys.plugins.survey.repository.SurveyRule.RuleType;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.web.repository.site.SiteManager;

/**
 * Abstract DAO for objects in the plugin survey.
 *
 */
public abstract class AbstractDAO extends AbstractLogEnabled implements Serviceable, Component, Contextualizable
{
    /** Ametys object resolver. */
    protected AmetysObjectResolver _resolver;
    /** The site manager */
    protected SiteManager _siteManager;
    /** Observer manager. */
    protected ObservationManager _observationManager;
    /** The current user provider. */
    protected CurrentUserProvider _currentUserProvider;
    /** Manager for retrieving uploaded files */
    protected UploadManager _uploadManager;
    /** JSON helper */
    protected JSONUtils _jsonUtils;
    
    /** The Avalon context */
    protected Context _context;
    /** The cocoon context */
    protected org.apache.cocoon.environment.Context _cocoonContext;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        _siteManager = (SiteManager) serviceManager.lookup(SiteManager.ROLE);
        _resolver = (AmetysObjectResolver) serviceManager.lookup(AmetysObjectResolver.ROLE);
        _observationManager = (ObservationManager) serviceManager.lookup(ObservationManager.ROLE);
        _currentUserProvider = (CurrentUserProvider) serviceManager.lookup(CurrentUserProvider.ROLE);
        _uploadManager = (UploadManager) serviceManager.lookup(UploadManager.ROLE);
        _jsonUtils = (JSONUtils) serviceManager.lookup(JSONUtils.ROLE);
    }
    
    @Override
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
        _cocoonContext = (org.apache.cocoon.environment.Context) context.get(Constants.CONTEXT_ENVIRONMENT_CONTEXT);
    }
    
    /**
     * Get the root node for surveys
     * @param siteName the site name
     * @param lang the language
     * @return the root node
     * @throws RepositoryException if an error occurs when manipulating the repository
     */
    public ModifiableTraversableAmetysObject getSurveyRootNode (String siteName, String lang) throws RepositoryException
    {
        ModifiableTraversableAmetysObject pluginsNode = _siteManager.getSite(siteName).getRootPlugins();
        
        ModifiableTraversableAmetysObject surveyNode = null;
        if (!pluginsNode.hasChild("survey"))
        {
            surveyNode = ((ModifiableTraversableAmetysObject) pluginsNode.createChild("survey", "ametys:unstructured")).createChild("ametys:surveys", "ametys:unstructured");
        }
        else
        {
            surveyNode = pluginsNode.getChild("survey/ametys:surveys");
        }
        
        if (!surveyNode.hasChild(lang))
        {
            surveyNode.createChild(lang, "ametys:unstructured");
            ((JCRAmetysObject) pluginsNode).getNode().getSession().save();
        }
        
        return pluginsNode.getChild("survey/ametys:surveys/" + lang);
    }
    
    /**
     * Provides the current user.
     * @return the user which cannot be <code>null</code>.
     */
    protected UserIdentity _getCurrentUser()
    {
        return _currentUserProvider.getUser();
    }
    
    /**
     * Update references after a survey copy
     * @param originalSurvey The original survey
     * @param createdSurvey The created survey
     */
    protected void updateReferencesAfterCopy (Survey originalSurvey, Survey createdSurvey)
    {
        for (SurveyPage cPage : createdSurvey.getPages())
        {
            updateReferencesAfterCopy (originalSurvey, createdSurvey, cPage);
        }
    }
    
    /**
     * Update references after copy
     * @param originalSurvey The original survey
     * @param createdSurvey The created survey
     * @param createdPage The created survey page
     */
    protected void updateReferencesAfterCopy (Survey originalSurvey, Survey createdSurvey, SurveyPage createdPage)
    {
        if (createdPage.hasRule())
        {
            SurveyRule rule = createdPage.getRule();
            String pageId = rule.getPage();
            if (pageId != null)
            {
                try
                {
                    // Find symmetric page in created survey
                    AmetysObject originalPage = _resolver.resolveById(pageId);
                    AmetysObject symmetricPage = createdSurvey.getChild(originalPage.getName());
                    createdPage.setRule(rule.getType(), symmetricPage.getId());
                }
                catch (UnknownAmetysObjectException e)
                {
                    // Delete the rule
                    new SLF4JLoggerAdapter(LoggerFactory.getLogger(this.getClass())).warn("Symmetric page has not found during copy. The rule is deleted.");
                    createdPage.deleteRule();
                }
            }
        }
        
        for (SurveyQuestion cQuestion : createdPage.getQuestions())
        {
            updateReferencesAfterCopy (originalSurvey, createdSurvey, cQuestion);
        }
    }
    
    /**
     * Update references after copy
     * @param originalSurvey The original survey
     * @param createdSurvey The created survey
     * @param createdQuestion The created survey question
     */
    protected void updateReferencesAfterCopy (Survey originalSurvey, Survey createdSurvey, SurveyQuestion createdQuestion)
    {
        for (SurveyRule cRule : createdQuestion.getRules())
        {
            String pageId = cRule.getPage();
            if (pageId != null)
            {
                String option = cRule.getOption();
                try
                {
                    // Find symmetric page in created survey
                    AmetysObject originalPage = _resolver.resolveById(pageId);
                    AmetysObject symmetricPage = createdSurvey.getChild(originalPage.getName());
                    
                    RuleType type = cRule.getType();
                    createdQuestion.deleteRule(option);
                    createdQuestion.addRules(option, type, symmetricPage.getId());
                }
                catch (UnknownAmetysObjectException e)
                {
                    // Delete rule
                    new SLF4JLoggerAdapter(LoggerFactory.getLogger(this.getClass())).warn("Symmetric page has not found during copy. The rule is deleted.");
                    createdQuestion.deleteRule(option);
                }
            }
        }
    }
    
    /**
     * Set the picture
     * @param elmt The survey element
     * @param valueAsStr The value as String
     */
    protected void setPicture (AbstractSurveyElement elmt, String valueAsStr)
    {
        if (StringUtils.isNotEmpty(valueAsStr))
        {
            Map<String, Object> picture = _jsonUtils.convertJsonToMap(valueAsStr);
            
            if (!picture.isEmpty())
            {
                String pictureType = (String) picture.get("type");
                String value = (String) picture.get("id");
                
                if (pictureType.equals("explorer") && !"untouched".equals(value))
                {
                    elmt.setResourcePicture(value);
                }
                else if (!"untouched".equals(value))
                {
                    UserIdentity user = _currentUserProvider.getUser();
                    Upload upload = _uploadManager.getUpload(user, value);
                    
                    String filename = upload.getFilename();
                    String mimeType = upload.getMimeType() != null ? upload.getMimeType() : _cocoonContext.getMimeType(filename);
                    String finalMimeType = mimeType != null ? mimeType : "application/unknown";
                    
                    elmt.setExternalPicture(finalMimeType, filename, upload.getInputStream());
                }
            }
            else
            {
                // Remove picture
                elmt.setNoPicture();
            }
            
        }
        else
        {
            // Remove picture
            elmt.setNoPicture();
        }
    }
    
    /**
     * Get the information about picture
     * @param elmt The survey element
     * @return The picture
     */
    protected Map<String, Object> getPictureInfo (AbstractSurveyElement elmt)
    {
        Map<String, Object> properties = new HashMap<>();
        
        properties.put("pictureAlternative", StringUtils.defaultString(elmt.getPictureAlternative()));
        
        Map<String, Object> pictureInfos = new HashMap<>();
        String pictureType = elmt.getPictureType();
        
        if (StringUtils.isNotBlank(pictureType))
        {
            if (pictureType.equals("resource"))
            {
                String resourceId = elmt.getResourcePictureId();
                pictureInfos.put("id", resourceId);
                try
                {
                    Resource resource = _resolver.resolveById(resourceId);
                    
                    pictureInfos.put("filename", resource.getName());
                    pictureInfos.put("size", resource.getLength());
                    pictureInfos.put("type", "explorer");
                    pictureInfos.put("lastModified", resource.getLastModified());
                    
                    String contextPath = ContextHelper.getRequest(_context).getContextPath();
                    String viewUrl = contextPath + "/plugins/explorer/resource?id=" + resource.getId();
                    String downloadUrl = viewUrl + "&download=true";
                    pictureInfos.put("viewUrl", viewUrl);
                    pictureInfos.put("downloadUrl", downloadUrl);
                }
                catch (UnknownAmetysObjectException e)
                {
                    getLogger().error("The resource of id'" + resourceId + "' does not exist anymore. The picture for element of id '" + elmt.getId() + "' will be ignored.", e);
                    properties.put("pictureNotFound", true);
                }
            }
            else if (pictureType.equals("external"))
            {
                Binary picMeta = elmt.getExternalPicture();
                
                pictureInfos.put("path", AbstractSurveyElement.PROPERTY_PICTURE);
                pictureInfos.put("filename", picMeta.getFilename());
                pictureInfos.put("size", picMeta.getLength());
                pictureInfos.put("lastModified", picMeta.getLastModificationDate());
                pictureInfos.put("type", "metadata");
                
                String contextPath = ContextHelper.getRequest(_context).getContextPath();
                String viewUrl = contextPath + "/plugins/cms/binary/" + AbstractSurveyElement.PROPERTY_PICTURE + "?objectId=" + elmt.getId();
                String downloadUrl = viewUrl + "&download=true";
                pictureInfos.put("viewUrl", viewUrl);
                pictureInfos.put("downloadUrl", downloadUrl);
                
            }
        }
        
        properties.put("picture", pictureInfos);
        
        return properties;
    }

}
