/*
 *  Copyright 2011 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.survey.data;

import java.sql.Connection;
import java.sql.SQLException;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.ibatis.session.SqlSession;

import org.ametys.core.datasource.AbstractMyBatisDAO;
import org.ametys.core.datasource.ConnectionHelper;
import org.ametys.core.user.UserIdentity;

/**
 * Survey answer DAO.
 */
public class SurveyAnswerDao extends AbstractMyBatisDAO
{
    /** The Avalon role name. */
    public static final String ROLE = SurveyAnswerDao.class.getName();
    
    /**
     * Get a session from its ID.
     * @param sessionId the session ID.
     * @return the session.
     */
    public SurveySession getSession(int sessionId)
    {
        Map<String, Object> params = new HashMap<>();
        params.put("id", sessionId);
        
        try (SqlSession session = getSession())
        {
            return (SurveySession) session.selectOne("SurveyAnswer.getSession", params);
        }
    }
    
    /**
     * Get a user session from the survey ID and the user identity.
     * @param surveyId the survey ID.
     * @param user the user.
     * @return the session.
     */
    public SurveySession getSession(String surveyId, UserIdentity user)
    {
        Map<String, Object> params = new HashMap<>();
        params.put("surveyId", surveyId);
        params.put("login", user.getLogin());
        params.put("population", user.getPopulationId());
        
        try (SqlSession sqlSession = getSession())
        {
            List<SurveySession> sessions = sqlSession.selectList("SurveyAnswer.getUserSession", params);
            SurveySession session = sessions != null && !sessions.isEmpty() ? sessions.get(0) : null;
            return session;
        }
    }
    
    /**
     * Get a session from its ID.
     * @param sessionId the session ID.
     * @return the session.
     */
    public SurveySession getSessionWithAnswers(int sessionId)
    {
        Map<String, Object> params = new HashMap<>();
        params.put("id", sessionId);
        
        try (SqlSession session = getSession())
        {
            return (SurveySession) session.selectOne("SurveyAnswer.getSessionWithAnswers", params);
        }
    }
    
    /**
     * Get the count of sessions for a given survey.
     * @param surveyId the survey ID.
     * @return the session count for this survey.
     */
    public int getSessionCount(String surveyId)
    {
        Map<String, Object> params = new HashMap<>();
        params.put("surveyId", surveyId);
        
        try (SqlSession session = getSession())
        {
            return (Integer) session.selectOne("SurveyAnswer.getSessionCount", params);
        }
    }
    
    /**
     * Get the session count list.
     * @return the session count list.
     */
    public List<Map<String, Object>> getAllSessionCount()
    {
        try (SqlSession session = getSession())
        {
            return session.selectList("SurveyAnswer.getAllSessionCount");
        }
    }
    
    /**
     * Get the exhaustive list of sessions.
     * @return the sessions.
     */
    public List<SurveySession> getSessions()
    {
        try (SqlSession session = getSession())
        {
            return session.selectList("SurveyAnswer.getSessions", Collections.EMPTY_MAP);
        }
    }
    
    /**
     * Get all the sessions of a given survey.
     * @param surveyId the survey ID.
     * @return the sessions.
     */
    public List<SurveySession> getSessions(String surveyId)
    {
        Map<String, Object> params = new HashMap<>();
        params.put("surveyId", surveyId);
        
        try (SqlSession session = getSession())
        {
            return session.selectList("SurveyAnswer.getSessions", params);
        }
    }
    
    /**
     * Get all the sessions of a given survey, with their answers.
     * @param surveyId the survey ID.
     * @return the sessions with their answers.
     */
    public List<SurveySession> getSessionsWithAnswers(String surveyId)
    {
        Map<String, Object> params = new HashMap<>();
        params.put("surveyId", surveyId);
        
        try (SqlSession session = getSession())
        {
            return session.selectList("SurveyAnswer.getSessionsWithAnswers", params);
        }
    }
    
    /**
     * Get the answers of a session.
     * @param sessionId the session ID.
     * @return the session answers.
     */
    public List<SurveyAnswer> getAnswers(int sessionId)
    {
        Map<String, Object> params = new HashMap<>();
        params.put("sessionId", sessionId);
        
        try (SqlSession session = getSession())
        {
            return session.selectList("SurveyAnswer.getAnswers", params);
        }
    }
    
    /**
     * Insert a session with its answers.
     * @param session the session.
     * @throws SQLException if a database access error occurs
     */
    public void addSession(SurveySession session) throws SQLException
    {
        try (SqlSession sqlSession = getSession())
        {
            try (Connection connection = sqlSession.getConnection())
            {
                String stmtId = "SurveyAnswer.addSession" + getStatementSuffix(connection);
                
                // Insert the session object and get the generated ID.
                sqlSession.insert(stmtId, session);
                
                int sessionId = session.getId();
                
                Map<String, Object> params = new HashMap<>();
                params.put("sessionId", sessionId);
                
                // Insert each answer.
                for (SurveyAnswer answer : session.getAnswers())
                {
                    params.put("questionId", answer.getQuestionId());
                    params.put("answer", answer.getValue());
                    
                    sqlSession.insert("SurveyAnswer.addAnswer", params);
                }
                
                // Commit the transaction.
                sqlSession.commit();
            }
        }
    }
    
    /**
     * Delete a session with its answers.
     * @param sessionId the session ID.
     */
    public void deleteSession(String sessionId)
    {
        try (SqlSession session = getSession())
        {
            // Delete the answers.
            session.delete("SurveyAnswer.deleteAnswers", sessionId);
            
            // Delete the session itself.
            session.delete("SurveyAnswer.deleteSession", sessionId);
            
            // Commit the transaction.
            session.commit();
        }
    }
    
    /**
     * Delete sessions related to a given survey.
     * @param surveyId the survey ID.
     */
    public void deleteSessions(String surveyId)
    {
        try (SqlSession session = getSession())
        {
            // Delete the answers.
            session.delete("SurveyAnswer.deleteSurveyAnswers", surveyId);
            
            // Delete the sessions.
            session.delete("SurveyAnswer.deleteSurveySessions", surveyId);
            
            // Commit the transaction.
            session.commit();
        }
    }
    
    /**
     * Get the statement name suffix, depending of the DBMS family.
     * @param connection the connection.
     * @return the statement suffix.
     */
    protected String getStatementSuffix(Connection connection)
    {
        String dbType = ConnectionHelper.getDatabaseType(connection);
        switch (dbType)
        {
            case ConnectionHelper.DATABASE_MYSQL:
                return "-mysql";
            case ConnectionHelper.DATABASE_POSTGRES:
                return "-postgresql";
            case ConnectionHelper.DATABASE_ORACLE:
                return "-oracle";
            case ConnectionHelper.DATABASE_DERBY:
                return "-derby";
            default:
                return "";
        }
    }
}
